/*
 * This file is part of veraPDF Validation for Arlington PDF Model, a module of the veraPDF project.
 * Copyright (c) 2015-2025, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF Validation for Arlington PDF Model is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF Validation for Arlington PDF Model as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF Validation for Arlington PDF Model as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.gf.model;

import org.verapdf.as.ASAtom;
import org.verapdf.cos.COSArray;
import org.verapdf.cos.COSObjType;
import org.verapdf.cos.COSObject;
import org.verapdf.extensions.ExtensionObjectType;
import org.verapdf.pd.PDCatalog;
import org.verapdf.pd.PDDocument;

import java.util.EnumSet;

public class GFAExtensions {

    public static EnumSet<ExtensionObjectType> getExtensions(PDDocument document, EnumSet<ExtensionObjectType> enableExtensions) {
        EnumSet<ExtensionObjectType> result = EnumSet.copyOf(enableExtensions);
        PDCatalog catalog = document.getCatalog();
        if (catalog == null) {
            return result;
        }
        COSObject extensions = catalog.getKey(ASAtom.EXTENSIONS);
        if (extensions == null) {
            return result;
        }
        COSObject iso = extensions.getKey(ASAtom.ISO_);
        if (iso != null) {
            if (iso.getType() == COSObjType.COS_DICT) {
                checkISOExtensions(iso, result);
            } else if (iso.getType() == COSObjType.COS_ARRAY) {
                for (COSObject entry : (COSArray)iso.getDirectBase()) {
                    if (entry.getType() == COSObjType.COS_DICT) {
                        checkISOExtensions(entry, result);
                    }
                }
            }
        }
        COSObject adbe = extensions.getKey(ASAtom.ADBE);
        if (adbe != null) {
            if (adbe.getType() == COSObjType.COS_DICT) {
                checkADBEExtensions(adbe, result);
            } else if (adbe.getType() == COSObjType.COS_ARRAY) {
                for (COSObject entry : (COSArray)adbe.getDirectBase()) {
                    if (entry.getType() == COSObjType.COS_DICT) {
                        checkADBEExtensions(entry, result);
                    }
                }
            }
        }
        return result;
    }

    private static void checkISOExtensions(COSObject isoDictionary, EnumSet<ExtensionObjectType> enableExtensions) {
        COSObject extensionLevel = isoDictionary.getKey(ASAtom.EXTENSION_LEVEL);
        if (extensionLevel.getType() == COSObjType.COS_INTEGER && extensionLevel.getInteger() == 24064) {
            enableExtensions.add(ExtensionObjectType.ISO_TS_24064);
        } else if (extensionLevel.getType() == COSObjType.COS_INTEGER && extensionLevel.getInteger() == 32001) {
            enableExtensions.add(ExtensionObjectType.ISO_TS_32001);
        } else if (extensionLevel.getType() == COSObjType.COS_INTEGER && extensionLevel.getInteger() == 32003) {
            enableExtensions.add(ExtensionObjectType.ISO_TS_32003);
        } else if (extensionLevel.getType() == COSObjType.COS_INTEGER && extensionLevel.getInteger() == 32004) {
            enableExtensions.add(ExtensionObjectType.ISO_TS_32004);
        } else if (extensionLevel.getType() == COSObjType.COS_INTEGER && extensionLevel.getInteger() == 32007) {
            enableExtensions.add(ExtensionObjectType.ISO_TS_32007);
        }
    }

    private static void checkADBEExtensions(COSObject isoDictionary, EnumSet<ExtensionObjectType> enableExtensions) {
        COSObject extensionLevel = isoDictionary.getKey(ASAtom.EXTENSION_LEVEL);
        if (extensionLevel.getType() == COSObjType.COS_INTEGER && extensionLevel.getInteger() == 3) {
            enableExtensions.add(ExtensionObjectType.ADBE_Extn3);
        }
    }
}
