/**
 * This file is part of veraPDF wcag algorithms, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF wcag algorithms is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF wcag algorithms as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF wcag algorithms as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.wcag.algorithms.entities.content;

import org.verapdf.wcag.algorithms.entities.IDocument;

import java.util.Map;
import java.util.HashMap;
import java.util.TreeSet;
import java.util.SortedSet;

public class LinesCollection {
	private final Map<Integer, SortedSet<LineChunk>> horizontalLines;
	private final Map<Integer, SortedSet<LineChunk>> verticalLines;

	private final IDocument document;

	public LinesCollection(IDocument document) {
		horizontalLines = new HashMap<>();
		verticalLines = new HashMap<>();
		this.document = document;
	}

	public SortedSet<LineChunk> getHorizontalLines(Integer pageNumber) {
		SortedSet<LineChunk> lines = horizontalLines.get(pageNumber);
		if (lines == null) {
			lines = parseHorizontalLines(pageNumber);
		}
		return lines;
	}

	private SortedSet<LineChunk> parseHorizontalLines(Integer pageNumber) {
		SortedSet<LineChunk> lines = new TreeSet<>(new LineChunk.HorizontalLineComparator());
		for (IChunk chunk : document.getArtifacts(pageNumber)) {
			if (chunk instanceof LineChunk) {
				LineChunk lineChunk = (LineChunk) chunk;
				if (lineChunk.isHorizontalLine()) {
					lines.add(lineChunk);
				}
			}
		}
		horizontalLines.put(pageNumber, lines);
		return lines;
	}

	public SortedSet<LineChunk> getVerticalLines(Integer pageNumber) {
		SortedSet<LineChunk> lines = verticalLines.get(pageNumber);
		if (lines == null) {
			lines = parseVerticalLines(pageNumber);
		}
		return lines;
	}

	private SortedSet<LineChunk> parseVerticalLines(Integer pageNumber) {
		SortedSet<LineChunk> lines = new TreeSet<>(new LineChunk.VerticalLineComparator());
		for (IChunk chunk : document.getArtifacts(pageNumber)) {
			if (chunk instanceof LineChunk) {
				LineChunk lineChunk = (LineChunk) chunk;
				if (lineChunk.isVerticalLine()) {
					lines.add(lineChunk);
				}
			}
		}
		verticalLines.put(pageNumber, lines);
		return lines;
	}
}
