/**
 * This file is part of veraPDF wcag algorithms, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF wcag algorithms is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF wcag algorithms as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF wcag algorithms as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.wcag.algorithms.entities.geometry;

import java.util.LinkedList;
import java.util.List;

public class MultiBoundingBox extends BoundingBox {

    private List<BoundingBox> boundingBoxes = new LinkedList<>();

    public MultiBoundingBox() {
        init(Double.MAX_VALUE, Double.MAX_VALUE, 0, 0);
        boundingBoxes.add(new BoundingBox());
    }

    public MultiBoundingBox(int pageNumber) {
        init(Double.MAX_VALUE, Double.MAX_VALUE, 0, 0);
        this.pageNumber = this.lastPageNumber = pageNumber;
        boundingBoxes.add(new BoundingBox(pageNumber));
    }

    public MultiBoundingBox(double[] bbox) {
        init(bbox[0], bbox[1], bbox[2], bbox[3]);
        boundingBoxes.add(new BoundingBox(bbox));
    }

    public MultiBoundingBox(int pageNumber, double[] bbox) {
        init(bbox[0], bbox[1], bbox[2], bbox[3]);
        this.pageNumber = this.lastPageNumber = pageNumber;
        boundingBoxes.add(new BoundingBox(pageNumber, bbox));
    }

    public MultiBoundingBox(int pageNumber, int lastPageNumber, double[] bbox) {
        init(bbox[0], bbox[1], bbox[2], bbox[3]);
        this.pageNumber = pageNumber;
        this.lastPageNumber = lastPageNumber;
        boundingBoxes.add(new BoundingBox(pageNumber, lastPageNumber, bbox));
    }

    public MultiBoundingBox(double left, double bottom, double right, double top) {
        init(left, bottom, right, top);
        boundingBoxes.add(new BoundingBox(left, bottom, right, top));
    }

    public MultiBoundingBox(int pageNumber, double left, double bottom, double right, double top) {
        init(left, bottom, right, top);
        this.pageNumber = this.lastPageNumber = pageNumber;
        boundingBoxes.add(new BoundingBox(pageNumber, left, bottom, right, top));
    }

    public MultiBoundingBox(int pageNumber, int lastPageNumber, double left, double bottom, double right, double top) {
        init(left, bottom, right, top);
        this.pageNumber = pageNumber;
        this.lastPageNumber = lastPageNumber;
        boundingBoxes.add(new BoundingBox(pageNumber, lastPageNumber, left, bottom, right, top));
    }

    public MultiBoundingBox(BoundingBox o) {
        init(o);
    }

    static public MultiBoundingBox union(BoundingBox first, BoundingBox second) {
        MultiBoundingBox result = new MultiBoundingBox(first);
        return result.union(second);
    }

    @Override
    public MultiBoundingBox union(BoundingBox second) {
        if (second == null || second.getPageNumber() == null) {
            return this;
        }
        if (pageNumber == null) {
            init(second);
            return this;
        }
        if (second instanceof MultiBoundingBox) {
            boundingBoxes.addAll(((MultiBoundingBox)second).boundingBoxes);
        } else {
            boundingBoxes.add(new BoundingBox(second));
        }

        if (second.leftX < this.leftX) {
            this.leftX = second.leftX;
        }
        if (this.rightX < second.rightX) {
            this.rightX = second.rightX;
        }
        if (this.pageNumber.equals(second.pageNumber)) {
            this.topY = Math.max(this.topY, second.topY);
        }
        else if (second.pageNumber < this.pageNumber) {
            this.topY = second.topY;
            this.pageNumber = second.pageNumber;
        }
        if (this.lastPageNumber.equals(second.lastPageNumber)) {
            this.bottomY = Math.min(this.bottomY, second.bottomY);
        }
        else if (this.lastPageNumber < second.lastPageNumber) {
            this.bottomY = second.bottomY;
            this.lastPageNumber = second.lastPageNumber;
        }
        return this;
    }

    //ToDo: cross methods

    @Override
    public BoundingBox scale(double coeffX, double coeffY) {
        for (BoundingBox boundingBox : boundingBoxes) {
            boundingBox.scale(coeffX, coeffY);
        }
        return super.scale(coeffX, coeffY);
    }

    @Override
    public BoundingBox move(double xShift, double yShift) {
        for (BoundingBox boundingBox : boundingBoxes) {
            boundingBox.move(xShift, yShift);
        }
        return super.move(xShift, yShift);
    }

    @Override
    public void init(BoundingBox rect) {
        super.init(rect);

        boundingBoxes = new LinkedList<>();
        if (rect instanceof MultiBoundingBox) {
            boundingBoxes.addAll(((MultiBoundingBox)rect).boundingBoxes);
        } else {
            boundingBoxes.add(new BoundingBox(rect));
        }
    }

    @Override
    public void init(double left, double bottom, double right, double top) {
        super.init(left, bottom, right, top);

        boundingBoxes = new LinkedList<>();
        boundingBoxes.add(new BoundingBox(this));
    }

    public List<BoundingBox> getBoundingBoxes() {
        return boundingBoxes;
    }
}
