/**
 * This file is part of veraPDF Validation Model API, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF Validation Model API is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF Validation Model API as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF Validation Model API as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.model;

import java.util.*;

/**
* This class represents names of superinterfaces and names of all properties for all generated interfaces.
*/
public final class ModelHelper {
	private final static Map<String, String> mapOfSuperNames = new HashMap<String, String>();
	private final static Map<String, List<String>> mapOfProperties = new HashMap<String, List<String>>();
	private final static Map<String, List<String>> mapOfLinks = new HashMap<String, List<String>>();
	private static List<String> properties;
	private static List<String> links;
	
	private ModelHelper(){
		
	}
	
	/**
	* @param objectName - the name of the object
	* @return List of supernames for the given object
	*/
	public static List<String> getListOfSuperNames(String objectName){
		List<String> res = new ArrayList<String>();
		
		String currentObject = mapOfSuperNames.get(objectName);
		
		while(currentObject != null){
			res.add(currentObject);
			currentObject = mapOfSuperNames.get(currentObject);
		}
		
		return res;
	}
	
	/**
	* @return Set of all type names
	*/
	public static Set<String> getTypes(){
		return mapOfSuperNames.keySet();
	}
	
	/**
	* @param objectName - the name of the object
	* @return List of names of properties for the given object
	*/
	public static List<String> getListOfProperties(String objectName){
		List<String> res = new ArrayList<String>();
		String currentObject = objectName;
		
		while(currentObject != null){
			for(String prop : mapOfProperties.get(currentObject)){
				res.add(prop);
			}
			
			currentObject = mapOfSuperNames.get(currentObject);
		}
		
		return res;
	}
	
	/**
	* @param objectName - the name of the object
	* @return List of names of links for the given object
	*/
	public static List<String> getListOfLinks(String objectName){
		List<String> res = new ArrayList<String>();
		
		String currentObject = objectName;
		
		while(currentObject != null){
			for(String link : mapOfLinks.get(currentObject)){
				res.add(link);
			}
			
			currentObject = mapOfSuperNames.get(currentObject);
		}
		
		return res;
	}


	private static void fillMapOfSuperNamesObject() {
		mapOfSuperNames.put("Object",null);
	}

	private static void fillMapOfPropertiesObject() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Object",properties);
	}

	private static void fillMapOfLinksObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("Object",links);
	}

	static {
		fillMapOfSuperNamesObject();
		fillMapOfPropertiesObject();
		fillMapOfLinksObject();
		}


	private static void fillMapOfSuperNamesExternal() {
		mapOfSuperNames.put("External","Object");
	}

	private static void fillMapOfPropertiesExternal() {
		properties = new ArrayList<String>();
		mapOfProperties.put("External",properties);
	}

	private static void fillMapOfLinksExternal() {
		links = new ArrayList<String>();
		mapOfLinks.put("External",links);
	}

	static {
		fillMapOfSuperNamesExternal();
		fillMapOfPropertiesExternal();
		fillMapOfLinksExternal();
	}


	private static void fillMapOfSuperNamesICCProfile() {
		mapOfSuperNames.put("ICCProfile","External");
	}

	private static void fillMapOfPropertiesICCProfile() {
		properties = new ArrayList<String>();
		properties.add("N");
		properties.add("deviceClass");
		properties.add("colorSpace");
		properties.add("version");
		properties.add("isValid");
		mapOfProperties.put("ICCProfile",properties);
	}

	private static void fillMapOfLinksICCProfile() {
		links = new ArrayList<String>();
		mapOfLinks.put("ICCProfile",links);
	}

	static {
		fillMapOfSuperNamesICCProfile();
		fillMapOfPropertiesICCProfile();
		fillMapOfLinksICCProfile();
	}


	private static void fillMapOfSuperNamesICCInputProfile() {
		mapOfSuperNames.put("ICCInputProfile","ICCProfile");
	}

	private static void fillMapOfPropertiesICCInputProfile() {
		properties = new ArrayList<String>();
		mapOfProperties.put("ICCInputProfile",properties);
	}

	private static void fillMapOfLinksICCInputProfile() {
		links = new ArrayList<String>();
		mapOfLinks.put("ICCInputProfile",links);
	}

	static {
		fillMapOfSuperNamesICCInputProfile();
		fillMapOfPropertiesICCInputProfile();
		fillMapOfLinksICCInputProfile();
	}


	private static void fillMapOfSuperNamesICCOutputProfile() {
		mapOfSuperNames.put("ICCOutputProfile","ICCProfile");
	}

	private static void fillMapOfPropertiesICCOutputProfile() {
		properties = new ArrayList<String>();
		properties.add("S");
		mapOfProperties.put("ICCOutputProfile",properties);
	}

	private static void fillMapOfLinksICCOutputProfile() {
		links = new ArrayList<String>();
		mapOfLinks.put("ICCOutputProfile",links);
	}

	static {
		fillMapOfSuperNamesICCOutputProfile();
		fillMapOfPropertiesICCOutputProfile();
		fillMapOfLinksICCOutputProfile();
	}


	private static void fillMapOfSuperNamesFontProgram() {
		mapOfSuperNames.put("FontProgram","External");
	}

	private static void fillMapOfPropertiesFontProgram() {
		properties = new ArrayList<String>();
		mapOfProperties.put("FontProgram",properties);
	}

	private static void fillMapOfLinksFontProgram() {
		links = new ArrayList<String>();
		mapOfLinks.put("FontProgram",links);
	}

	static {
		fillMapOfSuperNamesFontProgram();
		fillMapOfPropertiesFontProgram();
		fillMapOfLinksFontProgram();
	}


	private static void fillMapOfSuperNamesTrueTypeFontProgram() {
		mapOfSuperNames.put("TrueTypeFontProgram","FontProgram");
	}

	private static void fillMapOfPropertiesTrueTypeFontProgram() {
		properties = new ArrayList<String>();
		properties.add("isSymbolic");
		properties.add("nrCmaps");
		properties.add("cmap30Present");
		properties.add("cmap31Present");
		properties.add("cmap10Present");
		mapOfProperties.put("TrueTypeFontProgram",properties);
	}

	private static void fillMapOfLinksTrueTypeFontProgram() {
		links = new ArrayList<String>();
		mapOfLinks.put("TrueTypeFontProgram",links);
	}

	static {
		fillMapOfSuperNamesTrueTypeFontProgram();
		fillMapOfPropertiesTrueTypeFontProgram();
		fillMapOfLinksTrueTypeFontProgram();
	}


	private static void fillMapOfSuperNamesCMapFile() {
		mapOfSuperNames.put("CMapFile","External");
	}

	private static void fillMapOfPropertiesCMapFile() {
		properties = new ArrayList<String>();
		properties.add("WMode");
		properties.add("dictWMode");
		properties.add("maximalCID");
		mapOfProperties.put("CMapFile",properties);
	}

	private static void fillMapOfLinksCMapFile() {
		links = new ArrayList<String>();
		mapOfLinks.put("CMapFile",links);
	}

	static {
		fillMapOfSuperNamesCMapFile();
		fillMapOfPropertiesCMapFile();
		fillMapOfLinksCMapFile();
	}


	private static void fillMapOfSuperNamesJPEG2000() {
		mapOfSuperNames.put("JPEG2000","External");
	}

	private static void fillMapOfPropertiesJPEG2000() {
		properties = new ArrayList<String>();
		properties.add("nrColorChannels");
		properties.add("nrColorSpaceSpecs");
		properties.add("nrColorSpacesWithApproxField");
		properties.add("colrMethod");
		properties.add("colrEnumCS");
		properties.add("bitDepth");
		properties.add("bpccBoxPresent");
		properties.add("hasColorSpace");
		mapOfProperties.put("JPEG2000",properties);
	}

	private static void fillMapOfLinksJPEG2000() {
		links = new ArrayList<String>();
		mapOfLinks.put("JPEG2000",links);
	}

	static {
		fillMapOfSuperNamesJPEG2000();
		fillMapOfPropertiesJPEG2000();
		fillMapOfLinksJPEG2000();
	}


	private static void fillMapOfSuperNamesEmbeddedFile() {
		mapOfSuperNames.put("EmbeddedFile","External");
	}

	private static void fillMapOfPropertiesEmbeddedFile() {
		properties = new ArrayList<String>();
		properties.add("Subtype");
		properties.add("isValidPDFA12");
		properties.add("isValidPDFA124");
		mapOfProperties.put("EmbeddedFile",properties);
	}

	private static void fillMapOfLinksEmbeddedFile() {
		links = new ArrayList<String>();
		mapOfLinks.put("EmbeddedFile",links);
	}

	static {
		fillMapOfSuperNamesEmbeddedFile();
		fillMapOfPropertiesEmbeddedFile();
		fillMapOfLinksEmbeddedFile();
	}


	private static void fillMapOfSuperNamesPKCSDataObject() {
		mapOfSuperNames.put("PKCSDataObject","External");
	}

	private static void fillMapOfPropertiesPKCSDataObject() {
		properties = new ArrayList<String>();
		properties.add("SignerInfoCount");
		properties.add("signingCertificatePresent");
		mapOfProperties.put("PKCSDataObject",properties);
	}

	private static void fillMapOfLinksPKCSDataObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("PKCSDataObject",links);
	}

	static {
		fillMapOfSuperNamesPKCSDataObject();
		fillMapOfPropertiesPKCSDataObject();
		fillMapOfLinksPKCSDataObject();
		}


	private static void fillMapOfSuperNamesPDObject() {
		mapOfSuperNames.put("PDObject","Object");
	}

	private static void fillMapOfPropertiesPDObject() {
		properties = new ArrayList<String>();
		properties.add("entries");
		mapOfProperties.put("PDObject",properties);
	}

	private static void fillMapOfLinksPDObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDObject",links);
	}

	static {
		fillMapOfSuperNamesPDObject();
		fillMapOfPropertiesPDObject();
		fillMapOfLinksPDObject();
	}


	private static void fillMapOfSuperNamesPDDocument() {
		mapOfSuperNames.put("PDDocument","PDObject");
	}

	private static void fillMapOfPropertiesPDDocument() {
		properties = new ArrayList<String>();
		properties.add("containsAlternatePresentations");
		properties.add("validPDF");
		properties.add("containsAA");
		properties.add("outputColorSpace");
		properties.add("Version");
		properties.add("mostCommonOrientation");
		mapOfProperties.put("PDDocument",properties);
	}

	private static void fillMapOfLinksPDDocument() {
		links = new ArrayList<String>();
		links.add("outputIntents");
		links.add("pages");
		links.add("metadata");
		links.add("AcroForm");
		links.add("AA");
		links.add("OpenAction");
		links.add("Outlines");
		links.add("OCProperties");
		links.add("StructTreeRoot");
		links.add("Lang");
		links.add("Perms");
		mapOfLinks.put("PDDocument",links);
	}

	static {
		fillMapOfSuperNamesPDDocument();
		fillMapOfPropertiesPDDocument();
		fillMapOfLinksPDDocument();
	}


	private static void fillMapOfSuperNamesPDPage() {
		mapOfSuperNames.put("PDPage","PDObject");
	}

	private static void fillMapOfPropertiesPDPage() {
		properties = new ArrayList<String>();
		properties.add("containsPresSteps");
		properties.add("containsTransparency");
		properties.add("containsGroupCS");
		properties.add("containsAA");
		properties.add("Tabs");
		properties.add("orientation");
		properties.add("outputColorSpace");
		properties.add("pageNumber");
		mapOfProperties.put("PDPage",properties);
	}

	private static void fillMapOfLinksPDPage() {
		links = new ArrayList<String>();
		links.add("outputIntents");
		links.add("annots");
		links.add("AA");
		links.add("transparencyColorSpace");
		links.add("contentStream");
		links.add("parentTransparencyColorSpace");
		links.add("Group");
		links.add("MediaBox");
		links.add("CropBox");
		links.add("BleedBox");
		links.add("TrimBox");
		links.add("ArtBox");
		links.add("resources");
		mapOfLinks.put("PDPage",links);
	}

	static {
		fillMapOfSuperNamesPDPage();
		fillMapOfPropertiesPDPage();
		fillMapOfLinksPDPage();
	}


	private static void fillMapOfSuperNamesTransparencyColorSpace() {
		mapOfSuperNames.put("TransparencyColorSpace","Object");
	}

	private static void fillMapOfPropertiesTransparencyColorSpace() {
		properties = new ArrayList<String>();
		properties.add("colorSpaceType");
		mapOfProperties.put("TransparencyColorSpace",properties);
	}

	private static void fillMapOfLinksTransparencyColorSpace() {
		links = new ArrayList<String>();
		mapOfLinks.put("TransparencyColorSpace",links);
	}

	static {
		fillMapOfSuperNamesTransparencyColorSpace();
		fillMapOfPropertiesTransparencyColorSpace();
		fillMapOfLinksTransparencyColorSpace();
	}


	private static void fillMapOfSuperNamesOutputIntents() {
		mapOfSuperNames.put("OutputIntents","Object");
	}

	private static void fillMapOfPropertiesOutputIntents() {
		properties = new ArrayList<String>();
		properties.add("sameOutputProfileIndirect");
		properties.add("outputProfileIndirects");
		mapOfProperties.put("OutputIntents",properties);
	}

	private static void fillMapOfLinksOutputIntents() {
		links = new ArrayList<String>();
		links.add("outputIntents");
		mapOfLinks.put("OutputIntents",links);
	}

	static {
		fillMapOfSuperNamesOutputIntents();
		fillMapOfPropertiesOutputIntents();
		fillMapOfLinksOutputIntents();
	}


	private static void fillMapOfSuperNamesPDContentStream() {
		mapOfSuperNames.put("PDContentStream","PDObject");
	}

	private static void fillMapOfPropertiesPDContentStream() {
		properties = new ArrayList<String>();
		properties.add("undefinedResourceNames");
		properties.add("inheritedResourceNames");
		mapOfProperties.put("PDContentStream",properties);
	}

	private static void fillMapOfLinksPDContentStream() {
		links = new ArrayList<String>();
		links.add("operators");
		links.add("resources");
		mapOfLinks.put("PDContentStream",links);
	}

	static {
		fillMapOfSuperNamesPDContentStream();
		fillMapOfPropertiesPDContentStream();
		fillMapOfLinksPDContentStream();
	}


	private static void fillMapOfSuperNamesPDSemanticContentStream() {
		mapOfSuperNames.put("PDSemanticContentStream","PDContentStream");
	}

	private static void fillMapOfPropertiesPDSemanticContentStream() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDSemanticContentStream",properties);
	}

	private static void fillMapOfLinksPDSemanticContentStream() {
		links = new ArrayList<String>();
		links.add("content");
		mapOfLinks.put("PDSemanticContentStream",links);
	}

	static {
		fillMapOfSuperNamesPDSemanticContentStream();
		fillMapOfPropertiesPDSemanticContentStream();
		fillMapOfLinksPDSemanticContentStream();
	}


	private static void fillMapOfSuperNamesPDResource() {
		mapOfSuperNames.put("PDResource","PDObject");
	}

	private static void fillMapOfPropertiesPDResource() {
		properties = new ArrayList<String>();
		properties.add("isInherited");
		mapOfProperties.put("PDResource",properties);
	}

	private static void fillMapOfLinksPDResource() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDResource",links);
	}

	static {
		fillMapOfSuperNamesPDResource();
		fillMapOfPropertiesPDResource();
		fillMapOfLinksPDResource();
	}


	private static void fillMapOfSuperNamesPDColorSpace() {
		mapOfSuperNames.put("PDColorSpace","PDResource");
	}

	private static void fillMapOfPropertiesPDColorSpace() {
		properties = new ArrayList<String>();
		properties.add("nrComponents");
		mapOfProperties.put("PDColorSpace",properties);
	}

	private static void fillMapOfLinksPDColorSpace() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDColorSpace",links);
	}

	static {
		fillMapOfSuperNamesPDColorSpace();
		fillMapOfPropertiesPDColorSpace();
		fillMapOfLinksPDColorSpace();
	}


	private static void fillMapOfSuperNamesPDAdditionalActions() {
		mapOfSuperNames.put("PDAdditionalActions","PDObject");
	}

	private static void fillMapOfPropertiesPDAdditionalActions() {
		properties = new ArrayList<String>();
		properties.add("parentType");
		mapOfProperties.put("PDAdditionalActions",properties);
	}

	private static void fillMapOfLinksPDAdditionalActions() {
		links = new ArrayList<String>();
		links.add("Actions");
		mapOfLinks.put("PDAdditionalActions",links);
	}

	static {
		fillMapOfSuperNamesPDAdditionalActions();
		fillMapOfPropertiesPDAdditionalActions();
		fillMapOfLinksPDAdditionalActions();
	}


	private static void fillMapOfSuperNamesPDDeviceGray() {
		mapOfSuperNames.put("PDDeviceGray","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDDeviceGray() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDDeviceGray",properties);
	}

	private static void fillMapOfLinksPDDeviceGray() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDDeviceGray",links);
	}

	static {
		fillMapOfSuperNamesPDDeviceGray();
		fillMapOfPropertiesPDDeviceGray();
		fillMapOfLinksPDDeviceGray();
	}


	private static void fillMapOfSuperNamesPDDeviceRGB() {
		mapOfSuperNames.put("PDDeviceRGB","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDDeviceRGB() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDDeviceRGB",properties);
	}

	private static void fillMapOfLinksPDDeviceRGB() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDDeviceRGB",links);
	}

	static {
		fillMapOfSuperNamesPDDeviceRGB();
		fillMapOfPropertiesPDDeviceRGB();
		fillMapOfLinksPDDeviceRGB();
	}


	private static void fillMapOfSuperNamesPDDeviceCMYK() {
		mapOfSuperNames.put("PDDeviceCMYK","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDDeviceCMYK() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDDeviceCMYK",properties);
	}

	private static void fillMapOfLinksPDDeviceCMYK() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDDeviceCMYK",links);
	}

	static {
		fillMapOfSuperNamesPDDeviceCMYK();
		fillMapOfPropertiesPDDeviceCMYK();
		fillMapOfLinksPDDeviceCMYK();
	}


	private static void fillMapOfSuperNamesPDICCBased() {
		mapOfSuperNames.put("PDICCBased","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDICCBased() {
		properties = new ArrayList<String>();
		properties.add("ICCProfileIndirect");
		properties.add("currentTransparencyProfileIndirect");
		properties.add("ICCProfileMD5");
		properties.add("currentTransparencyICCProfileMD5");
		mapOfProperties.put("PDICCBased",properties);
	}

	private static void fillMapOfLinksPDICCBased() {
		links = new ArrayList<String>();
		links.add("iccProfile");
		mapOfLinks.put("PDICCBased",links);
	}

	static {
		fillMapOfSuperNamesPDICCBased();
		fillMapOfPropertiesPDICCBased();
		fillMapOfLinksPDICCBased();
	}


	private static void fillMapOfSuperNamesPDICCBasedCMYK() {
		mapOfSuperNames.put("PDICCBasedCMYK","PDICCBased");
	}

	private static void fillMapOfPropertiesPDICCBasedCMYK() {
		properties = new ArrayList<String>();
		properties.add("OPM");
		properties.add("overprintFlag");
		mapOfProperties.put("PDICCBasedCMYK",properties);
	}

	private static void fillMapOfLinksPDICCBasedCMYK() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDICCBasedCMYK",links);
	}

	static {
		fillMapOfSuperNamesPDICCBasedCMYK();
		fillMapOfPropertiesPDICCBasedCMYK();
		fillMapOfLinksPDICCBasedCMYK();
	}


	private static void fillMapOfSuperNamesPDLab() {
		mapOfSuperNames.put("PDLab","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDLab() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDLab",properties);
	}

	private static void fillMapOfLinksPDLab() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDLab",links);
	}

	static {
		fillMapOfSuperNamesPDLab();
		fillMapOfPropertiesPDLab();
		fillMapOfLinksPDLab();
	}


	private static void fillMapOfSuperNamesPDCalGray() {
		mapOfSuperNames.put("PDCalGray","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDCalGray() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDCalGray",properties);
	}

	private static void fillMapOfLinksPDCalGray() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDCalGray",links);
	}

	static {
		fillMapOfSuperNamesPDCalGray();
		fillMapOfPropertiesPDCalGray();
		fillMapOfLinksPDCalGray();
	}


	private static void fillMapOfSuperNamesPDCalRGB() {
		mapOfSuperNames.put("PDCalRGB","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDCalRGB() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDCalRGB",properties);
	}

	private static void fillMapOfLinksPDCalRGB() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDCalRGB",links);
	}

	static {
		fillMapOfSuperNamesPDCalRGB();
		fillMapOfPropertiesPDCalRGB();
		fillMapOfLinksPDCalRGB();
	}


	private static void fillMapOfSuperNamesPDSeparation() {
		mapOfSuperNames.put("PDSeparation","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDSeparation() {
		properties = new ArrayList<String>();
		properties.add("name");
		properties.add("areTintAndAlternateConsistent");
		mapOfProperties.put("PDSeparation",properties);
	}

	private static void fillMapOfLinksPDSeparation() {
		links = new ArrayList<String>();
		links.add("tintTransform");
		links.add("alternate");
		links.add("colorantName");
		mapOfLinks.put("PDSeparation",links);
	}

	static {
		fillMapOfSuperNamesPDSeparation();
		fillMapOfPropertiesPDSeparation();
		fillMapOfLinksPDSeparation();
	}


	private static void fillMapOfSuperNamesPDDeviceN() {
		mapOfSuperNames.put("PDDeviceN","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDDeviceN() {
		properties = new ArrayList<String>();
		properties.add("areColorantsPresent");
		mapOfProperties.put("PDDeviceN",properties);
	}

	private static void fillMapOfLinksPDDeviceN() {
		links = new ArrayList<String>();
		links.add("tintTransform");
		links.add("alternate");
		links.add("colorantNames");
		links.add("Colorants");
		links.add("processColor");
		mapOfLinks.put("PDDeviceN",links);
	}

	static {
		fillMapOfSuperNamesPDDeviceN();
		fillMapOfPropertiesPDDeviceN();
		fillMapOfLinksPDDeviceN();
	}


	private static void fillMapOfSuperNamesPDIndexed() {
		mapOfSuperNames.put("PDIndexed","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDIndexed() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDIndexed",properties);
	}

	private static void fillMapOfLinksPDIndexed() {
		links = new ArrayList<String>();
		links.add("base");
		mapOfLinks.put("PDIndexed",links);
	}

	static {
		fillMapOfSuperNamesPDIndexed();
		fillMapOfPropertiesPDIndexed();
		fillMapOfLinksPDIndexed();
	}


	private static void fillMapOfSuperNamesPDPattern() {
		mapOfSuperNames.put("PDPattern","PDColorSpace");
	}

	private static void fillMapOfPropertiesPDPattern() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDPattern",properties);
	}

	private static void fillMapOfLinksPDPattern() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDPattern",links);
	}

	static {
		fillMapOfSuperNamesPDPattern();
		fillMapOfPropertiesPDPattern();
		fillMapOfLinksPDPattern();
	}


	private static void fillMapOfSuperNamesPDTilingPattern() {
		mapOfSuperNames.put("PDTilingPattern","PDPattern");
	}

	private static void fillMapOfPropertiesPDTilingPattern() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDTilingPattern",properties);
	}

	private static void fillMapOfLinksPDTilingPattern() {
		links = new ArrayList<String>();
		links.add("contentStream");
		mapOfLinks.put("PDTilingPattern",links);
	}

	static {
		fillMapOfSuperNamesPDTilingPattern();
		fillMapOfPropertiesPDTilingPattern();
		fillMapOfLinksPDTilingPattern();
	}


	private static void fillMapOfSuperNamesPDShadingPattern() {
		mapOfSuperNames.put("PDShadingPattern","PDPattern");
	}

	private static void fillMapOfPropertiesPDShadingPattern() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDShadingPattern",properties);
	}

	private static void fillMapOfLinksPDShadingPattern() {
		links = new ArrayList<String>();
		links.add("shading");
		mapOfLinks.put("PDShadingPattern",links);
	}

	static {
		fillMapOfSuperNamesPDShadingPattern();
		fillMapOfPropertiesPDShadingPattern();
		fillMapOfLinksPDShadingPattern();
	}


	private static void fillMapOfSuperNamesPDFont() {
		mapOfSuperNames.put("PDFont","PDResource");
	}

	private static void fillMapOfPropertiesPDFont() {
		properties = new ArrayList<String>();
		properties.add("Type");
		properties.add("Subtype");
		properties.add("fontName");
		properties.add("isSymbolic");
		properties.add("renderingMode");
		properties.add("fontFileSubtype");
		properties.add("isItalic");
		mapOfProperties.put("PDFont",properties);
	}

	private static void fillMapOfLinksPDFont() {
		links = new ArrayList<String>();
		links.add("BaseFont");
		links.add("fontFile");
		mapOfLinks.put("PDFont",links);
	}

	static {
		fillMapOfSuperNamesPDFont();
		fillMapOfPropertiesPDFont();
		fillMapOfLinksPDFont();
	}


	private static void fillMapOfSuperNamesPDSimpleFont() {
		mapOfSuperNames.put("PDSimpleFont","PDFont");
	}

	private static void fillMapOfPropertiesPDSimpleFont() {
		properties = new ArrayList<String>();
		properties.add("isStandard");
		properties.add("FirstChar");
		properties.add("LastChar");
		properties.add("Widths_size");
		properties.add("Encoding");
		properties.add("containsDifferences");
		mapOfProperties.put("PDSimpleFont",properties);
	}

	private static void fillMapOfLinksPDSimpleFont() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDSimpleFont",links);
	}

	static {
		fillMapOfSuperNamesPDSimpleFont();
		fillMapOfPropertiesPDSimpleFont();
		fillMapOfLinksPDSimpleFont();
	}


	private static void fillMapOfSuperNamesPDTrueTypeFont() {
		mapOfSuperNames.put("PDTrueTypeFont","PDSimpleFont");
	}

	private static void fillMapOfPropertiesPDTrueTypeFont() {
		properties = new ArrayList<String>();
		properties.add("differencesAreUnicodeCompliant");
		mapOfProperties.put("PDTrueTypeFont",properties);
	}

	private static void fillMapOfLinksPDTrueTypeFont() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDTrueTypeFont",links);
	}

	static {
		fillMapOfSuperNamesPDTrueTypeFont();
		fillMapOfPropertiesPDTrueTypeFont();
		fillMapOfLinksPDTrueTypeFont();
	}


	private static void fillMapOfSuperNamesPDType1Font() {
		mapOfSuperNames.put("PDType1Font","PDSimpleFont");
	}

	private static void fillMapOfPropertiesPDType1Font() {
		properties = new ArrayList<String>();
		properties.add("CharSet");
		properties.add("charSetListsAllGlyphs");
		mapOfProperties.put("PDType1Font",properties);
	}

	private static void fillMapOfLinksPDType1Font() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDType1Font",links);
	}

	static {
		fillMapOfSuperNamesPDType1Font();
		fillMapOfPropertiesPDType1Font();
		fillMapOfLinksPDType1Font();
	}


	private static void fillMapOfSuperNamesPDType3Font() {
		mapOfSuperNames.put("PDType3Font","PDSimpleFont");
	}

	private static void fillMapOfPropertiesPDType3Font() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDType3Font",properties);
	}

	private static void fillMapOfLinksPDType3Font() {
		links = new ArrayList<String>();
		links.add("charStrings");
		mapOfLinks.put("PDType3Font",links);
	}

	static {
		fillMapOfSuperNamesPDType3Font();
		fillMapOfPropertiesPDType3Font();
		fillMapOfLinksPDType3Font();
	}


	private static void fillMapOfSuperNamesPDCIDFont() {
		mapOfSuperNames.put("PDCIDFont","PDFont");
	}

	private static void fillMapOfPropertiesPDCIDFont() {
		properties = new ArrayList<String>();
		properties.add("CIDToGIDMap");
		properties.add("cidSetListsAllGlyphs");
		mapOfProperties.put("PDCIDFont",properties);
	}

	private static void fillMapOfLinksPDCIDFont() {
		links = new ArrayList<String>();
		links.add("CIDSet");
		mapOfLinks.put("PDCIDFont",links);
	}

	static {
		fillMapOfSuperNamesPDCIDFont();
		fillMapOfPropertiesPDCIDFont();
		fillMapOfLinksPDCIDFont();
	}


	private static void fillMapOfSuperNamesPDType0Font() {
		mapOfSuperNames.put("PDType0Font","PDFont");
	}

	private static void fillMapOfPropertiesPDType0Font() {
		properties = new ArrayList<String>();
		properties.add("CIDFontOrdering");
		properties.add("CMapOrdering");
		properties.add("CIDFontRegistry");
		properties.add("CMapRegistry");
		properties.add("CIDFontSupplement");
		properties.add("CMapSupplement");
		properties.add("cmapName");
		mapOfProperties.put("PDType0Font",properties);
	}

	private static void fillMapOfLinksPDType0Font() {
		links = new ArrayList<String>();
		links.add("DescendantFonts");
		links.add("Encoding");
		mapOfLinks.put("PDType0Font",links);
	}

	static {
		fillMapOfSuperNamesPDType0Font();
		fillMapOfPropertiesPDType0Font();
		fillMapOfLinksPDType0Font();
	}


	private static void fillMapOfSuperNamesPDCMap() {
		mapOfSuperNames.put("PDCMap","PDObject");
	}

	private static void fillMapOfPropertiesPDCMap() {
		properties = new ArrayList<String>();
		properties.add("CMapName");
		mapOfProperties.put("PDCMap",properties);
	}

	private static void fillMapOfLinksPDCMap() {
		links = new ArrayList<String>();
		links.add("UseCMap");
		links.add("embeddedFile");
		mapOfLinks.put("PDCMap",links);
	}

	static {
		fillMapOfSuperNamesPDCMap();
		fillMapOfPropertiesPDCMap();
		fillMapOfLinksPDCMap();
	}


	private static void fillMapOfSuperNamesPDReferencedCMap() {
		mapOfSuperNames.put("PDReferencedCMap","PDCMap");
	}

	private static void fillMapOfPropertiesPDReferencedCMap() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDReferencedCMap",properties);
	}

	private static void fillMapOfLinksPDReferencedCMap() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDReferencedCMap",links);
	}

	static {
		fillMapOfSuperNamesPDReferencedCMap();
		fillMapOfPropertiesPDReferencedCMap();
		fillMapOfLinksPDReferencedCMap();
	}


	private static void fillMapOfSuperNamesPDXObject() {
		mapOfSuperNames.put("PDXObject","PDResource");
	}

	private static void fillMapOfPropertiesPDXObject() {
		properties = new ArrayList<String>();
		properties.add("Subtype");
		properties.add("containsOPI");
		properties.add("containsSMask");
		mapOfProperties.put("PDXObject",properties);
	}

	private static void fillMapOfLinksPDXObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDXObject",links);
	}

	static {
		fillMapOfSuperNamesPDXObject();
		fillMapOfPropertiesPDXObject();
		fillMapOfLinksPDXObject();
	}


	private static void fillMapOfSuperNamesPDXImage() {
		mapOfSuperNames.put("PDXImage","PDXObject");
	}

	private static void fillMapOfPropertiesPDXImage() {
		properties = new ArrayList<String>();
		properties.add("Interpolate");
		properties.add("containsAlternates");
		properties.add("BitsPerComponent");
		mapOfProperties.put("PDXImage",properties);
	}

	private static void fillMapOfLinksPDXImage() {
		links = new ArrayList<String>();
		links.add("imageCS");
		links.add("Alternates");
		links.add("Intent");
		links.add("jpxStream");
		links.add("SMask");
		mapOfLinks.put("PDXImage",links);
	}

	static {
		fillMapOfSuperNamesPDXImage();
		fillMapOfPropertiesPDXImage();
		fillMapOfLinksPDXImage();
	}


	private static void fillMapOfSuperNamesPDSMaskImage() {
		mapOfSuperNames.put("PDSMaskImage","PDXImage");
	}

	private static void fillMapOfPropertiesPDSMaskImage() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDSMaskImage",properties);
	}

	private static void fillMapOfLinksPDSMaskImage() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDSMaskImage",links);
	}

	static {
		fillMapOfSuperNamesPDSMaskImage();
		fillMapOfPropertiesPDSMaskImage();
		fillMapOfLinksPDSMaskImage();
	}


	private static void fillMapOfSuperNamesPDInlineImage() {
		mapOfSuperNames.put("PDInlineImage","PDXImage");
	}

	private static void fillMapOfPropertiesPDInlineImage() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDInlineImage",properties);
	}

	private static void fillMapOfLinksPDInlineImage() {
		links = new ArrayList<String>();
		links.add("F");
		mapOfLinks.put("PDInlineImage",links);
	}

	static {
		fillMapOfSuperNamesPDInlineImage();
		fillMapOfPropertiesPDInlineImage();
		fillMapOfLinksPDInlineImage();
	}


	private static void fillMapOfSuperNamesPDXForm() {
		mapOfSuperNames.put("PDXForm","PDXObject");
	}

	private static void fillMapOfPropertiesPDXForm() {
		properties = new ArrayList<String>();
		properties.add("Subtype2");
		properties.add("containsPS");
		properties.add("containsRef");
		properties.add("isUniqueSemanticParent");
		mapOfProperties.put("PDXForm",properties);
	}

	private static void fillMapOfLinksPDXForm() {
		links = new ArrayList<String>();
		links.add("transparencyColorSpace");
		links.add("contentStream");
		links.add("parentTransparencyColorSpace");
		links.add("Group");
		mapOfLinks.put("PDXForm",links);
	}

	static {
		fillMapOfSuperNamesPDXForm();
		fillMapOfPropertiesPDXForm();
		fillMapOfLinksPDXForm();
	}


	private static void fillMapOfSuperNamesPDResources() {
		mapOfSuperNames.put("PDResources","PDObject");
	}

	private static void fillMapOfPropertiesPDResources() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDResources",properties);
	}

	private static void fillMapOfLinksPDResources() {
		links = new ArrayList<String>();
		links.add("resourcesNames");
		mapOfLinks.put("PDResources",links);
	}

	static {
		fillMapOfSuperNamesPDResources();
		fillMapOfPropertiesPDResources();
		fillMapOfLinksPDResources();
	}


	private static void fillMapOfSuperNamesPDGroup() {
		mapOfSuperNames.put("PDGroup","PDObject");
	}

	private static void fillMapOfPropertiesPDGroup() {
		properties = new ArrayList<String>();
		properties.add("S");
		mapOfProperties.put("PDGroup",properties);
	}

	private static void fillMapOfLinksPDGroup() {
		links = new ArrayList<String>();
		links.add("colorSpace");
		mapOfLinks.put("PDGroup",links);
	}

	static {
		fillMapOfSuperNamesPDGroup();
		fillMapOfPropertiesPDGroup();
		fillMapOfLinksPDGroup();
	}


	private static void fillMapOfSuperNamesPDExtGState() {
		mapOfSuperNames.put("PDExtGState","PDResource");
	}

	private static void fillMapOfPropertiesPDExtGState() {
		properties = new ArrayList<String>();
		properties.add("TR");
		properties.add("TR2");
		properties.add("containsHTP");
		properties.add("containsHTO");
		properties.add("SMask");
		properties.add("BM");
		properties.add("ca");
		properties.add("CA");
		mapOfProperties.put("PDExtGState",properties);
	}

	private static void fillMapOfLinksPDExtGState() {
		links = new ArrayList<String>();
		links.add("HT");
		links.add("bm");
		links.add("RI");
		links.add("fontSize");
		links.add("customFunctions");
		mapOfLinks.put("PDExtGState",links);
	}

	static {
		fillMapOfSuperNamesPDExtGState();
		fillMapOfPropertiesPDExtGState();
		fillMapOfLinksPDExtGState();
	}


	private static void fillMapOfSuperNamesPDHalftone() {
		mapOfSuperNames.put("PDHalftone","PDObject");
	}

	private static void fillMapOfPropertiesPDHalftone() {
		properties = new ArrayList<String>();
		properties.add("HalftoneType");
		properties.add("HalftoneName");
		properties.add("colorantName");
		properties.add("TransferFunction");
		mapOfProperties.put("PDHalftone",properties);
	}

	private static void fillMapOfLinksPDHalftone() {
		links = new ArrayList<String>();
		links.add("customFunction");
		links.add("halftones");
		mapOfLinks.put("PDHalftone",links);
	}

	static {
		fillMapOfSuperNamesPDHalftone();
		fillMapOfPropertiesPDHalftone();
		fillMapOfLinksPDHalftone();
	}


	private static void fillMapOfSuperNamesPDShading() {
		mapOfSuperNames.put("PDShading","PDResource");
	}

	private static void fillMapOfPropertiesPDShading() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDShading",properties);
	}

	private static void fillMapOfLinksPDShading() {
		links = new ArrayList<String>();
		links.add("colorSpace");
		mapOfLinks.put("PDShading",links);
	}

	static {
		fillMapOfSuperNamesPDShading();
		fillMapOfPropertiesPDShading();
		fillMapOfLinksPDShading();
	}


	private static void fillMapOfSuperNamesPDAnnot() {
		mapOfSuperNames.put("PDAnnot","PDObject");
	}

	private static void fillMapOfPropertiesPDAnnot() {
		properties = new ArrayList<String>();
		properties.add("Subtype");
		properties.add("CA");
		properties.add("F");
		properties.add("AP");
		properties.add("FT");
		properties.add("N_type");
		properties.add("width");
		properties.add("height");
		properties.add("containsA");
		properties.add("containsAA");
		properties.add("structParentType");
		properties.add("Contents");
		properties.add("Alt");
		properties.add("isOutsideCropBox");
		mapOfProperties.put("PDAnnot",properties);
	}

	private static void fillMapOfLinksPDAnnot() {
		links = new ArrayList<String>();
		links.add("appearance");
		links.add("C");
		links.add("IC");
		links.add("A");
		links.add("AA");
		links.add("Lang");
		links.add("BM");
		mapOfLinks.put("PDAnnot",links);
	}

	static {
		fillMapOfSuperNamesPDAnnot();
		fillMapOfPropertiesPDAnnot();
		fillMapOfLinksPDAnnot();
	}


	private static void fillMapOfSuperNamesPD3DAnnot() {
		mapOfSuperNames.put("PD3DAnnot","PDAnnot");
	}

	private static void fillMapOfPropertiesPD3DAnnot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PD3DAnnot",properties);
	}

	private static void fillMapOfLinksPD3DAnnot() {
		links = new ArrayList<String>();
		links.add("stream3D");
		mapOfLinks.put("PD3DAnnot",links);
	}

	static {
		fillMapOfSuperNamesPD3DAnnot();
		fillMapOfPropertiesPD3DAnnot();
		fillMapOfLinksPD3DAnnot();
	}


	private static void fillMapOfSuperNamesPDWidgetAnnot() {
		mapOfSuperNames.put("PDWidgetAnnot","PDAnnot");
	}

	private static void fillMapOfPropertiesPDWidgetAnnot() {
		properties = new ArrayList<String>();
		properties.add("TU");
		mapOfProperties.put("PDWidgetAnnot",properties);
	}

	private static void fillMapOfLinksPDWidgetAnnot() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDWidgetAnnot",links);
	}

	static {
		fillMapOfSuperNamesPDWidgetAnnot();
		fillMapOfPropertiesPDWidgetAnnot();
		fillMapOfLinksPDWidgetAnnot();
	}


	private static void fillMapOfSuperNamesPDPrinterMarkAnnot() {
		mapOfSuperNames.put("PDPrinterMarkAnnot","PDAnnot");
	}

	private static void fillMapOfPropertiesPDPrinterMarkAnnot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDPrinterMarkAnnot",properties);
	}

	private static void fillMapOfLinksPDPrinterMarkAnnot() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDPrinterMarkAnnot",links);
	}

	static {
		fillMapOfSuperNamesPDPrinterMarkAnnot();
		fillMapOfPropertiesPDPrinterMarkAnnot();
		fillMapOfLinksPDPrinterMarkAnnot();
	}


	private static void fillMapOfSuperNamesPDLinkAnnot() {
		mapOfSuperNames.put("PDLinkAnnot","PDAnnot");
	}

	private static void fillMapOfPropertiesPDLinkAnnot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDLinkAnnot",properties);
	}

	private static void fillMapOfLinksPDLinkAnnot() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDLinkAnnot",links);
	}

	static {
		fillMapOfSuperNamesPDLinkAnnot();
		fillMapOfPropertiesPDLinkAnnot();
		fillMapOfLinksPDLinkAnnot();
	}


	private static void fillMapOfSuperNamesPDTrapNetAnnot() {
		mapOfSuperNames.put("PDTrapNetAnnot","PDAnnot");
	}

	private static void fillMapOfPropertiesPDTrapNetAnnot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDTrapNetAnnot",properties);
	}

	private static void fillMapOfLinksPDTrapNetAnnot() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDTrapNetAnnot",links);
	}

	static {
		fillMapOfSuperNamesPDTrapNetAnnot();
		fillMapOfPropertiesPDTrapNetAnnot();
		fillMapOfLinksPDTrapNetAnnot();
	}


	private static void fillMapOfSuperNamesPD3DStream() {
		mapOfSuperNames.put("PD3DStream","PDObject");
	}

	private static void fillMapOfPropertiesPD3DStream() {
		properties = new ArrayList<String>();
		properties.add("Subtype");
		mapOfProperties.put("PD3DStream",properties);
	}

	private static void fillMapOfLinksPD3DStream() {
		links = new ArrayList<String>();
		links.add("colorSpace");
		mapOfLinks.put("PD3DStream",links);
	}

	static {
		fillMapOfSuperNamesPD3DStream();
		fillMapOfPropertiesPD3DStream();
		fillMapOfLinksPD3DStream();
	}


	private static void fillMapOfSuperNamesPDAcroForm() {
		mapOfSuperNames.put("PDAcroForm","PDObject");
	}

	private static void fillMapOfPropertiesPDAcroForm() {
		properties = new ArrayList<String>();
		properties.add("NeedAppearances");
		properties.add("containsXFA");
		properties.add("dynamicRender");
		mapOfProperties.put("PDAcroForm",properties);
	}

	private static void fillMapOfLinksPDAcroForm() {
		links = new ArrayList<String>();
		links.add("formFields");
		mapOfLinks.put("PDAcroForm",links);
	}

	static {
		fillMapOfSuperNamesPDAcroForm();
		fillMapOfPropertiesPDAcroForm();
		fillMapOfLinksPDAcroForm();
	}


	private static void fillMapOfSuperNamesPDFormField() {
		mapOfSuperNames.put("PDFormField","PDObject");
	}

	private static void fillMapOfPropertiesPDFormField() {
		properties = new ArrayList<String>();
		properties.add("FT");
		properties.add("containsAA");
		properties.add("TU");
		mapOfProperties.put("PDFormField",properties);
	}

	private static void fillMapOfLinksPDFormField() {
		links = new ArrayList<String>();
		links.add("AA");
		links.add("Lang");
		mapOfLinks.put("PDFormField",links);
	}

	static {
		fillMapOfSuperNamesPDFormField();
		fillMapOfPropertiesPDFormField();
		fillMapOfLinksPDFormField();
	}


	private static void fillMapOfSuperNamesPDSignatureField() {
		mapOfSuperNames.put("PDSignatureField","PDFormField");
	}

	private static void fillMapOfPropertiesPDSignatureField() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDSignatureField",properties);
	}

	private static void fillMapOfLinksPDSignatureField() {
		links = new ArrayList<String>();
		links.add("V");
		mapOfLinks.put("PDSignatureField",links);
	}

	static {
		fillMapOfSuperNamesPDSignatureField();
		fillMapOfPropertiesPDSignatureField();
		fillMapOfLinksPDSignatureField();
	}


	private static void fillMapOfSuperNamesPDSignature() {
		mapOfSuperNames.put("PDSignature","PDObject");
	}

	private static void fillMapOfPropertiesPDSignature() {
		properties = new ArrayList<String>();
		properties.add("doesByteRangeCoverEntireDocument");
		mapOfProperties.put("PDSignature",properties);
	}

	private static void fillMapOfLinksPDSignature() {
		links = new ArrayList<String>();
		links.add("Contents");
		links.add("Reference");
		mapOfLinks.put("PDSignature",links);
	}

	static {
		fillMapOfSuperNamesPDSignature();
		fillMapOfPropertiesPDSignature();
		fillMapOfLinksPDSignature();
	}


	private static void fillMapOfSuperNamesPDAction() {
		mapOfSuperNames.put("PDAction","PDObject");
	}

	private static void fillMapOfPropertiesPDAction() {
		properties = new ArrayList<String>();
		properties.add("S");
		mapOfProperties.put("PDAction",properties);
	}

	private static void fillMapOfLinksPDAction() {
		links = new ArrayList<String>();
		links.add("Next");
		mapOfLinks.put("PDAction",links);
	}

	static {
		fillMapOfSuperNamesPDAction();
		fillMapOfPropertiesPDAction();
		fillMapOfLinksPDAction();
	}


	private static void fillMapOfSuperNamesPDRenditionAction() {
		mapOfSuperNames.put("PDRenditionAction","PDAction");
	}

	private static void fillMapOfPropertiesPDRenditionAction() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDRenditionAction",properties);
	}

	private static void fillMapOfLinksPDRenditionAction() {
		links = new ArrayList<String>();
		links.add("C");
		mapOfLinks.put("PDRenditionAction",links);
	}

	static {
		fillMapOfSuperNamesPDRenditionAction();
		fillMapOfPropertiesPDRenditionAction();
		fillMapOfLinksPDRenditionAction();
	}


	private static void fillMapOfSuperNamesPDEncryption() {
		mapOfSuperNames.put("PDEncryption","PDObject");
	}

	private static void fillMapOfPropertiesPDEncryption() {
		properties = new ArrayList<String>();
		properties.add("P");
		mapOfProperties.put("PDEncryption",properties);
	}

	private static void fillMapOfLinksPDEncryption() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDEncryption",links);
	}

	static {
		fillMapOfSuperNamesPDEncryption();
		fillMapOfPropertiesPDEncryption();
		fillMapOfLinksPDEncryption();
	}


	private static void fillMapOfSuperNamesPDMediaClip() {
		mapOfSuperNames.put("PDMediaClip","PDObject");
	}

	private static void fillMapOfPropertiesPDMediaClip() {
		properties = new ArrayList<String>();
		properties.add("CT");
		properties.add("Alt");
		properties.add("hasCorrectAlt");
		mapOfProperties.put("PDMediaClip",properties);
	}

	private static void fillMapOfLinksPDMediaClip() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDMediaClip",links);
	}

	static {
		fillMapOfSuperNamesPDMediaClip();
		fillMapOfPropertiesPDMediaClip();
		fillMapOfLinksPDMediaClip();
	}


	private static void fillMapOfSuperNamesPDNamedAction() {
		mapOfSuperNames.put("PDNamedAction","PDAction");
	}

	private static void fillMapOfPropertiesPDNamedAction() {
		properties = new ArrayList<String>();
		properties.add("N");
		mapOfProperties.put("PDNamedAction",properties);
	}

	private static void fillMapOfLinksPDNamedAction() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDNamedAction",links);
	}

	static {
		fillMapOfSuperNamesPDNamedAction();
		fillMapOfPropertiesPDNamedAction();
		fillMapOfLinksPDNamedAction();
	}


	private static void fillMapOfSuperNamesPDGoToAction() {
		mapOfSuperNames.put("PDGoToAction","PDAction");
	}

	private static void fillMapOfPropertiesPDGoToAction() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDGoToAction",properties);
	}

	private static void fillMapOfLinksPDGoToAction() {
		links = new ArrayList<String>();
		links.add("D");
		mapOfLinks.put("PDGoToAction",links);
	}

	static {
		fillMapOfSuperNamesPDGoToAction();
		fillMapOfPropertiesPDGoToAction();
		fillMapOfLinksPDGoToAction();
	}


	private static void fillMapOfSuperNamesPDGoToRemoteAction() {
		mapOfSuperNames.put("PDGoToRemoteAction","PDGoToAction");
	}

	private static void fillMapOfPropertiesPDGoToRemoteAction() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDGoToRemoteAction",properties);
	}

	private static void fillMapOfLinksPDGoToRemoteAction() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDGoToRemoteAction",links);
	}

	static {
		fillMapOfSuperNamesPDGoToRemoteAction();
		fillMapOfPropertiesPDGoToRemoteAction();
		fillMapOfLinksPDGoToRemoteAction();
	}


	private static void fillMapOfSuperNamesPDMetadata() {
		mapOfSuperNames.put("PDMetadata","PDObject");
	}

	private static void fillMapOfPropertiesPDMetadata() {
		properties = new ArrayList<String>();
		properties.add("Filter");
		mapOfProperties.put("PDMetadata",properties);
	}

	private static void fillMapOfLinksPDMetadata() {
		links = new ArrayList<String>();
		links.add("XMPPackage");
		links.add("stream");
		mapOfLinks.put("PDMetadata",links);
	}

	static {
		fillMapOfSuperNamesPDMetadata();
		fillMapOfPropertiesPDMetadata();
		fillMapOfLinksPDMetadata();
	}


	private static void fillMapOfSuperNamesPDOutputIntent() {
		mapOfSuperNames.put("PDOutputIntent","PDObject");
	}

	private static void fillMapOfPropertiesPDOutputIntent() {
		properties = new ArrayList<String>();
		properties.add("destOutputProfileIndirect");
		properties.add("containsDestOutputProfileRef");
		properties.add("OutputConditionIdentifier");
		properties.add("S");
		properties.add("ICCProfileMD5");
		mapOfProperties.put("PDOutputIntent",properties);
	}

	private static void fillMapOfLinksPDOutputIntent() {
		links = new ArrayList<String>();
		links.add("destProfile");
		mapOfLinks.put("PDOutputIntent",links);
	}

	static {
		fillMapOfSuperNamesPDOutputIntent();
		fillMapOfPropertiesPDOutputIntent();
		fillMapOfLinksPDOutputIntent();
	}


	private static void fillMapOfSuperNamesPDOCProperties() {
		mapOfSuperNames.put("PDOCProperties","PDObject");
	}

	private static void fillMapOfPropertiesPDOCProperties() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDOCProperties",properties);
	}

	private static void fillMapOfLinksPDOCProperties() {
		links = new ArrayList<String>();
		links.add("D");
		links.add("Configs");
		mapOfLinks.put("PDOCProperties",links);
	}

	static {
		fillMapOfSuperNamesPDOCProperties();
		fillMapOfPropertiesPDOCProperties();
		fillMapOfLinksPDOCProperties();
	}


	private static void fillMapOfSuperNamesPDOCConfig() {
		mapOfSuperNames.put("PDOCConfig","PDObject");
	}

	private static void fillMapOfPropertiesPDOCConfig() {
		properties = new ArrayList<String>();
		properties.add("Name");
		properties.add("hasDuplicateName");
		properties.add("AS");
		properties.add("OCGsNotContainedInOrder");
		mapOfProperties.put("PDOCConfig",properties);
	}

	private static void fillMapOfLinksPDOCConfig() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDOCConfig",links);
	}

	static {
		fillMapOfSuperNamesPDOCConfig();
		fillMapOfPropertiesPDOCConfig();
		fillMapOfLinksPDOCConfig();
	}


	private static void fillMapOfSuperNamesPDOutline() {
		mapOfSuperNames.put("PDOutline","PDObject");
	}

	private static void fillMapOfPropertiesPDOutline() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDOutline",properties);
	}

	private static void fillMapOfLinksPDOutline() {
		links = new ArrayList<String>();
		links.add("A");
		mapOfLinks.put("PDOutline",links);
	}

	static {
		fillMapOfSuperNamesPDOutline();
		fillMapOfPropertiesPDOutline();
		fillMapOfLinksPDOutline();
	}


	private static void fillMapOfSuperNamesPDStructTreeRoot() {
		mapOfSuperNames.put("PDStructTreeRoot","PDObject");
	}

	private static void fillMapOfPropertiesPDStructTreeRoot() {
		properties = new ArrayList<String>();
		properties.add("topLevelFirstElementStandardType");
		properties.add("kidsStandardTypes");
		properties.add("hasContentItems");
		mapOfProperties.put("PDStructTreeRoot",properties);
	}

	private static void fillMapOfLinksPDStructTreeRoot() {
		links = new ArrayList<String>();
		links.add("K");
		links.add("roleMapNames");
		mapOfLinks.put("PDStructTreeRoot",links);
	}

	static {
		fillMapOfSuperNamesPDStructTreeRoot();
		fillMapOfPropertiesPDStructTreeRoot();
		fillMapOfLinksPDStructTreeRoot();
	}


	private static void fillMapOfSuperNamesPDStructElem() {
		mapOfSuperNames.put("PDStructElem","PDObject");
	}

	private static void fillMapOfPropertiesPDStructElem() {
		properties = new ArrayList<String>();
		properties.add("Type");
		properties.add("kidsStandardTypes");
		properties.add("parentStandardType");
		properties.add("hasContentItems");
		properties.add("valueS");
		properties.add("standardType");
		properties.add("parentLang");
		properties.add("isRemappedStandardType");
		properties.add("Alt");
		properties.add("ActualText");
		properties.add("E");
		properties.add("circularMappingExist");
		mapOfProperties.put("PDStructElem",properties);
	}

	private static void fillMapOfLinksPDStructElem() {
		links = new ArrayList<String>();
		links.add("K");
		links.add("S");
		links.add("Lang");
		links.add("actualText");
		mapOfLinks.put("PDStructElem",links);
	}

	static {
		fillMapOfSuperNamesPDStructElem();
		fillMapOfPropertiesPDStructElem();
		fillMapOfLinksPDStructElem();
	}


	private static void fillMapOfSuperNamesPDPerms() {
		mapOfSuperNames.put("PDPerms","PDObject");
	}

	private static void fillMapOfPropertiesPDPerms() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDPerms",properties);
	}

	private static void fillMapOfLinksPDPerms() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDPerms",links);
	}

	static {
		fillMapOfSuperNamesPDPerms();
		fillMapOfPropertiesPDPerms();
		fillMapOfLinksPDPerms();
	}


	private static void fillMapOfSuperNamesPDSigRef() {
		mapOfSuperNames.put("PDSigRef","PDObject");
	}

	private static void fillMapOfPropertiesPDSigRef() {
		properties = new ArrayList<String>();
		properties.add("permsContainDocMDP");
		mapOfProperties.put("PDSigRef",properties);
	}

	private static void fillMapOfLinksPDSigRef() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDSigRef",links);
	}

	static {
		fillMapOfSuperNamesPDSigRef();
		fillMapOfPropertiesPDSigRef();
		fillMapOfLinksPDSigRef();
	}


	private static void fillMapOfSuperNamesPDFunction() {
		mapOfSuperNames.put("PDFunction","PDObject");
	}

	private static void fillMapOfPropertiesPDFunction() {
		properties = new ArrayList<String>();
		properties.add("FunctionType");
		mapOfProperties.put("PDFunction",properties);
	}

	private static void fillMapOfLinksPDFunction() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDFunction",links);
	}

	static {
		fillMapOfSuperNamesPDFunction();
		fillMapOfPropertiesPDFunction();
		fillMapOfLinksPDFunction();
	}


	private static void fillMapOfSuperNamesPDType0Function() {
		mapOfSuperNames.put("PDType0Function","PDFunction");
	}

	private static void fillMapOfPropertiesPDType0Function() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDType0Function",properties);
	}

	private static void fillMapOfLinksPDType0Function() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDType0Function",links);
	}

	static {
		fillMapOfSuperNamesPDType0Function();
		fillMapOfPropertiesPDType0Function();
		fillMapOfLinksPDType0Function();
	}


	private static void fillMapOfSuperNamesPDType2Function() {
		mapOfSuperNames.put("PDType2Function","PDFunction");
	}

	private static void fillMapOfPropertiesPDType2Function() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDType2Function",properties);
	}

	private static void fillMapOfLinksPDType2Function() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDType2Function",links);
	}

	static {
		fillMapOfSuperNamesPDType2Function();
		fillMapOfPropertiesPDType2Function();
		fillMapOfLinksPDType2Function();
	}


	private static void fillMapOfSuperNamesPDType3Function() {
		mapOfSuperNames.put("PDType3Function","PDFunction");
	}

	private static void fillMapOfPropertiesPDType3Function() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDType3Function",properties);
	}

	private static void fillMapOfLinksPDType3Function() {
		links = new ArrayList<String>();
		links.add("Functions");
		mapOfLinks.put("PDType3Function",links);
	}

	static {
		fillMapOfSuperNamesPDType3Function();
		fillMapOfPropertiesPDType3Function();
		fillMapOfLinksPDType3Function();
	}


	private static void fillMapOfSuperNamesPDType4Function() {
		mapOfSuperNames.put("PDType4Function","PDFunction");
	}

	private static void fillMapOfPropertiesPDType4Function() {
		properties = new ArrayList<String>();
		mapOfProperties.put("PDType4Function",properties);
	}

	private static void fillMapOfLinksPDType4Function() {
		links = new ArrayList<String>();
		links.add("operators");
		mapOfLinks.put("PDType4Function",links);
	}

	static {
		fillMapOfSuperNamesPDType4Function();
		fillMapOfPropertiesPDType4Function();
		fillMapOfLinksPDType4Function();
		}


	private static void fillMapOfSuperNamesXMPObject() {
		mapOfSuperNames.put("XMPObject","Object");
	}

	private static void fillMapOfPropertiesXMPObject() {
		properties = new ArrayList<String>();
		mapOfProperties.put("XMPObject",properties);
	}

	private static void fillMapOfLinksXMPObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("XMPObject",links);
	}

	static {
		fillMapOfSuperNamesXMPObject();
		fillMapOfPropertiesXMPObject();
		fillMapOfLinksXMPObject();
	}


	private static void fillMapOfSuperNamesXMPPackage() {
		mapOfSuperNames.put("XMPPackage","XMPObject");
	}

	private static void fillMapOfPropertiesXMPPackage() {
		properties = new ArrayList<String>();
		properties.add("isSerializationValid");
		properties.add("actualEncoding");
		properties.add("bytes");
		properties.add("encoding");
		mapOfProperties.put("XMPPackage",properties);
	}

	private static void fillMapOfLinksXMPPackage() {
		links = new ArrayList<String>();
		links.add("Properties");
		links.add("ExtensionSchemasContainers");
		mapOfLinks.put("XMPPackage",links);
	}

	static {
		fillMapOfSuperNamesXMPPackage();
		fillMapOfPropertiesXMPPackage();
		fillMapOfLinksXMPPackage();
	}


	private static void fillMapOfSuperNamesMainXMPPackage() {
		mapOfSuperNames.put("MainXMPPackage","XMPPackage");
	}

	private static void fillMapOfPropertiesMainXMPPackage() {
		properties = new ArrayList<String>();
		properties.add("dc_title");
		mapOfProperties.put("MainXMPPackage",properties);
	}

	private static void fillMapOfLinksMainXMPPackage() {
		links = new ArrayList<String>();
		links.add("Identification");
		links.add("UAIdentification");
		mapOfLinks.put("MainXMPPackage",links);
	}

	static {
		fillMapOfSuperNamesMainXMPPackage();
		fillMapOfPropertiesMainXMPPackage();
		fillMapOfLinksMainXMPPackage();
	}


	private static void fillMapOfSuperNamesXMPProperty() {
		mapOfSuperNames.put("XMPProperty","XMPObject");
	}

	private static void fillMapOfPropertiesXMPProperty() {
		properties = new ArrayList<String>();
		properties.add("isPredefinedInXMP2004");
		properties.add("isPredefinedInXMP2005");
		properties.add("isDefinedInCurrentPackage");
		properties.add("isDefinedInMainPackage");
		properties.add("isValueTypeCorrect");
		properties.add("predefinedType");
		mapOfProperties.put("XMPProperty",properties);
	}

	private static void fillMapOfLinksXMPProperty() {
		links = new ArrayList<String>();
		mapOfLinks.put("XMPProperty",links);
	}

	static {
		fillMapOfSuperNamesXMPProperty();
		fillMapOfPropertiesXMPProperty();
		fillMapOfLinksXMPProperty();
	}


	private static void fillMapOfSuperNamesXMPMMHistoryProperty() {
		mapOfSuperNames.put("XMPMMHistoryProperty","XMPProperty");
	}

	private static void fillMapOfPropertiesXMPMMHistoryProperty() {
		properties = new ArrayList<String>();
		mapOfProperties.put("XMPMMHistoryProperty",properties);
	}

	private static void fillMapOfLinksXMPMMHistoryProperty() {
		links = new ArrayList<String>();
		links.add("ResourceEvents");
		mapOfLinks.put("XMPMMHistoryProperty",links);
	}

	static {
		fillMapOfSuperNamesXMPMMHistoryProperty();
		fillMapOfPropertiesXMPMMHistoryProperty();
		fillMapOfLinksXMPMMHistoryProperty();
	}


	private static void fillMapOfSuperNamesXMPLangAlt() {
		mapOfSuperNames.put("XMPLangAlt","XMPProperty");
	}

	private static void fillMapOfPropertiesXMPLangAlt() {
		properties = new ArrayList<String>();
		properties.add("xDefault");
		mapOfProperties.put("XMPLangAlt",properties);
	}

	private static void fillMapOfLinksXMPLangAlt() {
		links = new ArrayList<String>();
		mapOfLinks.put("XMPLangAlt",links);
	}

	static {
		fillMapOfSuperNamesXMPLangAlt();
		fillMapOfPropertiesXMPLangAlt();
		fillMapOfLinksXMPLangAlt();
	}


	private static void fillMapOfSuperNamesXMPMMHistoryResourceEvent() {
		mapOfSuperNames.put("XMPMMHistoryResourceEvent","XMPObject");
	}

	private static void fillMapOfPropertiesXMPMMHistoryResourceEvent() {
		properties = new ArrayList<String>();
		properties.add("action");
		properties.add("parameters");
		properties.add("when");
		mapOfProperties.put("XMPMMHistoryResourceEvent",properties);
	}

	private static void fillMapOfLinksXMPMMHistoryResourceEvent() {
		links = new ArrayList<String>();
		mapOfLinks.put("XMPMMHistoryResourceEvent",links);
	}

	static {
		fillMapOfSuperNamesXMPMMHistoryResourceEvent();
		fillMapOfPropertiesXMPMMHistoryResourceEvent();
		fillMapOfLinksXMPMMHistoryResourceEvent();
	}


	private static void fillMapOfSuperNamesPDFAIdentification() {
		mapOfSuperNames.put("PDFAIdentification","XMPObject");
	}

	private static void fillMapOfPropertiesPDFAIdentification() {
		properties = new ArrayList<String>();
		properties.add("part");
		properties.add("conformance");
		properties.add("rev");
		properties.add("partPrefix");
		properties.add("conformancePrefix");
		properties.add("amdPrefix");
		properties.add("corrPrefix");
		properties.add("revPrefix");
		mapOfProperties.put("PDFAIdentification",properties);
	}

	private static void fillMapOfLinksPDFAIdentification() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDFAIdentification",links);
	}

	static {
		fillMapOfSuperNamesPDFAIdentification();
		fillMapOfPropertiesPDFAIdentification();
		fillMapOfLinksPDFAIdentification();
	}


	private static void fillMapOfSuperNamesPDFUAIdentification() {
		mapOfSuperNames.put("PDFUAIdentification","XMPObject");
	}

	private static void fillMapOfPropertiesPDFUAIdentification() {
		properties = new ArrayList<String>();
		properties.add("part");
		mapOfProperties.put("PDFUAIdentification",properties);
	}

	private static void fillMapOfLinksPDFUAIdentification() {
		links = new ArrayList<String>();
		mapOfLinks.put("PDFUAIdentification",links);
	}

	static {
		fillMapOfSuperNamesPDFUAIdentification();
		fillMapOfPropertiesPDFUAIdentification();
		fillMapOfLinksPDFUAIdentification();
	}


	private static void fillMapOfSuperNamesExtensionSchemaObject() {
		mapOfSuperNames.put("ExtensionSchemaObject","XMPObject");
	}

	private static void fillMapOfPropertiesExtensionSchemaObject() {
		properties = new ArrayList<String>();
		properties.add("containsUndefinedFields");
		properties.add("undefinedFields");
		mapOfProperties.put("ExtensionSchemaObject",properties);
	}

	private static void fillMapOfLinksExtensionSchemaObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("ExtensionSchemaObject",links);
	}

	static {
		fillMapOfSuperNamesExtensionSchemaObject();
		fillMapOfPropertiesExtensionSchemaObject();
		fillMapOfLinksExtensionSchemaObject();
	}


	private static void fillMapOfSuperNamesExtensionSchemasContainer() {
		mapOfSuperNames.put("ExtensionSchemasContainer","XMPObject");
	}

	private static void fillMapOfPropertiesExtensionSchemasContainer() {
		properties = new ArrayList<String>();
		properties.add("prefix");
		properties.add("isValidBag");
		mapOfProperties.put("ExtensionSchemasContainer",properties);
	}

	private static void fillMapOfLinksExtensionSchemasContainer() {
		links = new ArrayList<String>();
		links.add("ExtensionSchemaDefinitions");
		mapOfLinks.put("ExtensionSchemasContainer",links);
	}

	static {
		fillMapOfSuperNamesExtensionSchemasContainer();
		fillMapOfPropertiesExtensionSchemasContainer();
		fillMapOfLinksExtensionSchemasContainer();
	}


	private static void fillMapOfSuperNamesExtensionSchemaDefinition() {
		mapOfSuperNames.put("ExtensionSchemaDefinition","ExtensionSchemaObject");
	}

	private static void fillMapOfPropertiesExtensionSchemaDefinition() {
		properties = new ArrayList<String>();
		properties.add("isNamespaceURIValidURI");
		properties.add("isPrefixValidText");
		properties.add("isPropertyValidSeq");
		properties.add("isSchemaValidText");
		properties.add("isValueTypeValidSeq");
		properties.add("namespaceURIPrefix");
		properties.add("prefixPrefix");
		properties.add("propertyPrefix");
		properties.add("schemaPrefix");
		properties.add("valueTypePrefix");
		mapOfProperties.put("ExtensionSchemaDefinition",properties);
	}

	private static void fillMapOfLinksExtensionSchemaDefinition() {
		links = new ArrayList<String>();
		links.add("ExtensionSchemaProperties");
		links.add("ExtensionSchemaValueTypes");
		mapOfLinks.put("ExtensionSchemaDefinition",links);
	}

	static {
		fillMapOfSuperNamesExtensionSchemaDefinition();
		fillMapOfPropertiesExtensionSchemaDefinition();
		fillMapOfLinksExtensionSchemaDefinition();
	}


	private static void fillMapOfSuperNamesExtensionSchemaProperty() {
		mapOfSuperNames.put("ExtensionSchemaProperty","ExtensionSchemaObject");
	}

	private static void fillMapOfPropertiesExtensionSchemaProperty() {
		properties = new ArrayList<String>();
		properties.add("category");
		properties.add("isCategoryValidText");
		properties.add("isDescriptionValidText");
		properties.add("isNameValidText");
		properties.add("isValueTypeValidText");
		properties.add("isValueTypeDefined");
		properties.add("categoryPrefix");
		properties.add("descriptionPrefix");
		properties.add("namePrefix");
		properties.add("valueTypePrefix");
		mapOfProperties.put("ExtensionSchemaProperty",properties);
	}

	private static void fillMapOfLinksExtensionSchemaProperty() {
		links = new ArrayList<String>();
		mapOfLinks.put("ExtensionSchemaProperty",links);
	}

	static {
		fillMapOfSuperNamesExtensionSchemaProperty();
		fillMapOfPropertiesExtensionSchemaProperty();
		fillMapOfLinksExtensionSchemaProperty();
	}


	private static void fillMapOfSuperNamesExtensionSchemaValueType() {
		mapOfSuperNames.put("ExtensionSchemaValueType","ExtensionSchemaObject");
	}

	private static void fillMapOfPropertiesExtensionSchemaValueType() {
		properties = new ArrayList<String>();
		properties.add("isDescriptionValidText");
		properties.add("isFieldValidSeq");
		properties.add("isNamespaceURIValidURI");
		properties.add("isPrefixValidText");
		properties.add("isTypeValidText");
		properties.add("descriptionPrefix");
		properties.add("fieldPrefix");
		properties.add("namespaceURIPrefix");
		properties.add("prefixPrefix");
		properties.add("typePrefix");
		mapOfProperties.put("ExtensionSchemaValueType",properties);
	}

	private static void fillMapOfLinksExtensionSchemaValueType() {
		links = new ArrayList<String>();
		links.add("ExtensionSchemaFields");
		mapOfLinks.put("ExtensionSchemaValueType",links);
	}

	static {
		fillMapOfSuperNamesExtensionSchemaValueType();
		fillMapOfPropertiesExtensionSchemaValueType();
		fillMapOfLinksExtensionSchemaValueType();
	}


	private static void fillMapOfSuperNamesExtensionSchemaField() {
		mapOfSuperNames.put("ExtensionSchemaField","ExtensionSchemaObject");
	}

	private static void fillMapOfPropertiesExtensionSchemaField() {
		properties = new ArrayList<String>();
		properties.add("isDescriptionValidText");
		properties.add("isNameValidText");
		properties.add("isValueTypeValidText");
		properties.add("isValueTypeDefined");
		properties.add("descriptionPrefix");
		properties.add("namePrefix");
		properties.add("valueTypePrefix");
		mapOfProperties.put("ExtensionSchemaField",properties);
	}

	private static void fillMapOfLinksExtensionSchemaField() {
		links = new ArrayList<String>();
		mapOfLinks.put("ExtensionSchemaField",links);
	}

	static {
		fillMapOfSuperNamesExtensionSchemaField();
		fillMapOfPropertiesExtensionSchemaField();
		fillMapOfLinksExtensionSchemaField();
		}


	private static void fillMapOfSuperNamesOperator() {
		mapOfSuperNames.put("Operator","Object");
	}

	private static void fillMapOfPropertiesOperator() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Operator",properties);
	}

	private static void fillMapOfLinksOperator() {
		links = new ArrayList<String>();
		mapOfLinks.put("Operator",links);
	}

	static {
		fillMapOfSuperNamesOperator();
		fillMapOfPropertiesOperator();
		fillMapOfLinksOperator();
	}


	private static void fillMapOfSuperNamesOpGeneralGS() {
		mapOfSuperNames.put("OpGeneralGS","Operator");
	}

	private static void fillMapOfPropertiesOpGeneralGS() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpGeneralGS",properties);
	}

	private static void fillMapOfLinksOpGeneralGS() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpGeneralGS",links);
	}

	static {
		fillMapOfSuperNamesOpGeneralGS();
		fillMapOfPropertiesOpGeneralGS();
		fillMapOfLinksOpGeneralGS();
	}


	private static void fillMapOfSuperNamesOp_w_line_width() {
		mapOfSuperNames.put("Op_w_line_width","OpGeneralGS");
	}

	private static void fillMapOfPropertiesOp_w_line_width() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_w_line_width",properties);
	}

	private static void fillMapOfLinksOp_w_line_width() {
		links = new ArrayList<String>();
		links.add("lineWidth");
		mapOfLinks.put("Op_w_line_width",links);
	}

	static {
		fillMapOfSuperNamesOp_w_line_width();
		fillMapOfPropertiesOp_w_line_width();
		fillMapOfLinksOp_w_line_width();
	}


	private static void fillMapOfSuperNamesOp_J_line_cap() {
		mapOfSuperNames.put("Op_J_line_cap","OpGeneralGS");
	}

	private static void fillMapOfPropertiesOp_J_line_cap() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_J_line_cap",properties);
	}

	private static void fillMapOfLinksOp_J_line_cap() {
		links = new ArrayList<String>();
		links.add("lineCap");
		mapOfLinks.put("Op_J_line_cap",links);
	}

	static {
		fillMapOfSuperNamesOp_J_line_cap();
		fillMapOfPropertiesOp_J_line_cap();
		fillMapOfLinksOp_J_line_cap();
	}


	private static void fillMapOfSuperNamesOp_j_line_join() {
		mapOfSuperNames.put("Op_j_line_join","OpGeneralGS");
	}

	private static void fillMapOfPropertiesOp_j_line_join() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_j_line_join",properties);
	}

	private static void fillMapOfLinksOp_j_line_join() {
		links = new ArrayList<String>();
		links.add("lineJoin");
		mapOfLinks.put("Op_j_line_join",links);
	}

	static {
		fillMapOfSuperNamesOp_j_line_join();
		fillMapOfPropertiesOp_j_line_join();
		fillMapOfLinksOp_j_line_join();
	}


	private static void fillMapOfSuperNamesOp_M_miter_limit() {
		mapOfSuperNames.put("Op_M_miter_limit","OpGeneralGS");
	}

	private static void fillMapOfPropertiesOp_M_miter_limit() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_M_miter_limit",properties);
	}

	private static void fillMapOfLinksOp_M_miter_limit() {
		links = new ArrayList<String>();
		links.add("miterLimit");
		mapOfLinks.put("Op_M_miter_limit",links);
	}

	static {
		fillMapOfSuperNamesOp_M_miter_limit();
		fillMapOfPropertiesOp_M_miter_limit();
		fillMapOfLinksOp_M_miter_limit();
	}


	private static void fillMapOfSuperNamesOp_d() {
		mapOfSuperNames.put("Op_d","OpGeneralGS");
	}

	private static void fillMapOfPropertiesOp_d() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_d",properties);
	}

	private static void fillMapOfLinksOp_d() {
		links = new ArrayList<String>();
		links.add("dashArray");
		links.add("dashPhase");
		mapOfLinks.put("Op_d",links);
	}

	static {
		fillMapOfSuperNamesOp_d();
		fillMapOfPropertiesOp_d();
		fillMapOfLinksOp_d();
	}


	private static void fillMapOfSuperNamesOp_ri() {
		mapOfSuperNames.put("Op_ri","OpGeneralGS");
	}

	private static void fillMapOfPropertiesOp_ri() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_ri",properties);
	}

	private static void fillMapOfLinksOp_ri() {
		links = new ArrayList<String>();
		links.add("renderingIntent");
		mapOfLinks.put("Op_ri",links);
	}

	static {
		fillMapOfSuperNamesOp_ri();
		fillMapOfPropertiesOp_ri();
		fillMapOfLinksOp_ri();
	}


	private static void fillMapOfSuperNamesOp_i() {
		mapOfSuperNames.put("Op_i","OpGeneralGS");
	}

	private static void fillMapOfPropertiesOp_i() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_i",properties);
	}

	private static void fillMapOfLinksOp_i() {
		links = new ArrayList<String>();
		links.add("flatness");
		mapOfLinks.put("Op_i",links);
	}

	static {
		fillMapOfSuperNamesOp_i();
		fillMapOfPropertiesOp_i();
		fillMapOfLinksOp_i();
	}


	private static void fillMapOfSuperNamesOp_gs() {
		mapOfSuperNames.put("Op_gs","OpGeneralGS");
	}

	private static void fillMapOfPropertiesOp_gs() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_gs",properties);
	}

	private static void fillMapOfLinksOp_gs() {
		links = new ArrayList<String>();
		links.add("extGState");
		mapOfLinks.put("Op_gs",links);
	}

	static {
		fillMapOfSuperNamesOp_gs();
		fillMapOfPropertiesOp_gs();
		fillMapOfLinksOp_gs();
	}


	private static void fillMapOfSuperNamesOpSpecialGS() {
		mapOfSuperNames.put("OpSpecialGS","Operator");
	}

	private static void fillMapOfPropertiesOpSpecialGS() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpSpecialGS",properties);
	}

	private static void fillMapOfLinksOpSpecialGS() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpSpecialGS",links);
	}

	static {
		fillMapOfSuperNamesOpSpecialGS();
		fillMapOfPropertiesOpSpecialGS();
		fillMapOfLinksOpSpecialGS();
	}


	private static void fillMapOfSuperNamesOp_q_gsave() {
		mapOfSuperNames.put("Op_q_gsave","OpSpecialGS");
	}

	private static void fillMapOfPropertiesOp_q_gsave() {
		properties = new ArrayList<String>();
		properties.add("nestingLevel");
		mapOfProperties.put("Op_q_gsave",properties);
	}

	private static void fillMapOfLinksOp_q_gsave() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_q_gsave",links);
	}

	static {
		fillMapOfSuperNamesOp_q_gsave();
		fillMapOfPropertiesOp_q_gsave();
		fillMapOfLinksOp_q_gsave();
	}


	private static void fillMapOfSuperNamesOp_Q_grestore() {
		mapOfSuperNames.put("Op_Q_grestore","OpSpecialGS");
	}

	private static void fillMapOfPropertiesOp_Q_grestore() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Q_grestore",properties);
	}

	private static void fillMapOfLinksOp_Q_grestore() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_Q_grestore",links);
	}

	static {
		fillMapOfSuperNamesOp_Q_grestore();
		fillMapOfPropertiesOp_Q_grestore();
		fillMapOfLinksOp_Q_grestore();
	}


	private static void fillMapOfSuperNamesOp_cm() {
		mapOfSuperNames.put("Op_cm","OpSpecialGS");
	}

	private static void fillMapOfPropertiesOp_cm() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_cm",properties);
	}

	private static void fillMapOfLinksOp_cm() {
		links = new ArrayList<String>();
		links.add("matrix");
		mapOfLinks.put("Op_cm",links);
	}

	static {
		fillMapOfSuperNamesOp_cm();
		fillMapOfPropertiesOp_cm();
		fillMapOfLinksOp_cm();
	}


	private static void fillMapOfSuperNamesOpPathConstruction() {
		mapOfSuperNames.put("OpPathConstruction","Operator");
	}

	private static void fillMapOfPropertiesOpPathConstruction() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpPathConstruction",properties);
	}

	private static void fillMapOfLinksOpPathConstruction() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpPathConstruction",links);
	}

	static {
		fillMapOfSuperNamesOpPathConstruction();
		fillMapOfPropertiesOpPathConstruction();
		fillMapOfLinksOpPathConstruction();
	}


	private static void fillMapOfSuperNamesOp_m_moveto() {
		mapOfSuperNames.put("Op_m_moveto","OpPathConstruction");
	}

	private static void fillMapOfPropertiesOp_m_moveto() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_m_moveto",properties);
	}

	private static void fillMapOfLinksOp_m_moveto() {
		links = new ArrayList<String>();
		links.add("point");
		mapOfLinks.put("Op_m_moveto",links);
	}

	static {
		fillMapOfSuperNamesOp_m_moveto();
		fillMapOfPropertiesOp_m_moveto();
		fillMapOfLinksOp_m_moveto();
	}


	private static void fillMapOfSuperNamesOp_l() {
		mapOfSuperNames.put("Op_l","OpPathConstruction");
	}

	private static void fillMapOfPropertiesOp_l() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_l",properties);
	}

	private static void fillMapOfLinksOp_l() {
		links = new ArrayList<String>();
		links.add("point");
		mapOfLinks.put("Op_l",links);
	}

	static {
		fillMapOfSuperNamesOp_l();
		fillMapOfPropertiesOp_l();
		fillMapOfLinksOp_l();
	}


	private static void fillMapOfSuperNamesOp_c() {
		mapOfSuperNames.put("Op_c","OpPathConstruction");
	}

	private static void fillMapOfPropertiesOp_c() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_c",properties);
	}

	private static void fillMapOfLinksOp_c() {
		links = new ArrayList<String>();
		links.add("controlPoints");
		mapOfLinks.put("Op_c",links);
	}

	static {
		fillMapOfSuperNamesOp_c();
		fillMapOfPropertiesOp_c();
		fillMapOfLinksOp_c();
	}


	private static void fillMapOfSuperNamesOp_v() {
		mapOfSuperNames.put("Op_v","OpPathConstruction");
	}

	private static void fillMapOfPropertiesOp_v() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_v",properties);
	}

	private static void fillMapOfLinksOp_v() {
		links = new ArrayList<String>();
		links.add("controlPoints");
		mapOfLinks.put("Op_v",links);
	}

	static {
		fillMapOfSuperNamesOp_v();
		fillMapOfPropertiesOp_v();
		fillMapOfLinksOp_v();
	}


	private static void fillMapOfSuperNamesOp_y() {
		mapOfSuperNames.put("Op_y","OpPathConstruction");
	}

	private static void fillMapOfPropertiesOp_y() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_y",properties);
	}

	private static void fillMapOfLinksOp_y() {
		links = new ArrayList<String>();
		links.add("controlPoints");
		mapOfLinks.put("Op_y",links);
	}

	static {
		fillMapOfSuperNamesOp_y();
		fillMapOfPropertiesOp_y();
		fillMapOfLinksOp_y();
	}


	private static void fillMapOfSuperNamesOp_h() {
		mapOfSuperNames.put("Op_h","OpPathConstruction");
	}

	private static void fillMapOfPropertiesOp_h() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_h",properties);
	}

	private static void fillMapOfLinksOp_h() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_h",links);
	}

	static {
		fillMapOfSuperNamesOp_h();
		fillMapOfPropertiesOp_h();
		fillMapOfLinksOp_h();
	}


	private static void fillMapOfSuperNamesOp_re() {
		mapOfSuperNames.put("Op_re","OpPathConstruction");
	}

	private static void fillMapOfPropertiesOp_re() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_re",properties);
	}

	private static void fillMapOfLinksOp_re() {
		links = new ArrayList<String>();
		links.add("rectBox");
		mapOfLinks.put("Op_re",links);
	}

	static {
		fillMapOfSuperNamesOp_re();
		fillMapOfPropertiesOp_re();
		fillMapOfLinksOp_re();
	}


	private static void fillMapOfSuperNamesOpPathPaint() {
		mapOfSuperNames.put("OpPathPaint","Operator");
	}

	private static void fillMapOfPropertiesOpPathPaint() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpPathPaint",properties);
	}

	private static void fillMapOfLinksOpPathPaint() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpPathPaint",links);
	}

	static {
		fillMapOfSuperNamesOpPathPaint();
		fillMapOfPropertiesOpPathPaint();
		fillMapOfLinksOpPathPaint();
	}


	private static void fillMapOfSuperNamesOp_s_close_stroke() {
		mapOfSuperNames.put("Op_s_close_stroke","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_s_close_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_s_close_stroke",properties);
	}

	private static void fillMapOfLinksOp_s_close_stroke() {
		links = new ArrayList<String>();
		links.add("strokeCS");
		mapOfLinks.put("Op_s_close_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_s_close_stroke();
		fillMapOfPropertiesOp_s_close_stroke();
		fillMapOfLinksOp_s_close_stroke();
	}


	private static void fillMapOfSuperNamesOp_S_stroke() {
		mapOfSuperNames.put("Op_S_stroke","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_S_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_S_stroke",properties);
	}

	private static void fillMapOfLinksOp_S_stroke() {
		links = new ArrayList<String>();
		links.add("strokeCS");
		mapOfLinks.put("Op_S_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_S_stroke();
		fillMapOfPropertiesOp_S_stroke();
		fillMapOfLinksOp_S_stroke();
	}


	private static void fillMapOfSuperNamesOp_f_fill() {
		mapOfSuperNames.put("Op_f_fill","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_f_fill() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_f_fill",properties);
	}

	private static void fillMapOfLinksOp_f_fill() {
		links = new ArrayList<String>();
		links.add("fillCS");
		mapOfLinks.put("Op_f_fill",links);
	}

	static {
		fillMapOfSuperNamesOp_f_fill();
		fillMapOfPropertiesOp_f_fill();
		fillMapOfLinksOp_f_fill();
	}


	private static void fillMapOfSuperNamesOp_F_fill_obsolete() {
		mapOfSuperNames.put("Op_F_fill_obsolete","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_F_fill_obsolete() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_F_fill_obsolete",properties);
	}

	private static void fillMapOfLinksOp_F_fill_obsolete() {
		links = new ArrayList<String>();
		links.add("fillCS");
		mapOfLinks.put("Op_F_fill_obsolete",links);
	}

	static {
		fillMapOfSuperNamesOp_F_fill_obsolete();
		fillMapOfPropertiesOp_F_fill_obsolete();
		fillMapOfLinksOp_F_fill_obsolete();
	}


	private static void fillMapOfSuperNamesOp_FStar() {
		mapOfSuperNames.put("Op_FStar","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_FStar() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_FStar",properties);
	}

	private static void fillMapOfLinksOp_FStar() {
		links = new ArrayList<String>();
		links.add("fillCS");
		mapOfLinks.put("Op_FStar",links);
	}

	static {
		fillMapOfSuperNamesOp_FStar();
		fillMapOfPropertiesOp_FStar();
		fillMapOfLinksOp_FStar();
	}


	private static void fillMapOfSuperNamesOp_B_fill_stroke() {
		mapOfSuperNames.put("Op_B_fill_stroke","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_B_fill_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_B_fill_stroke",properties);
	}

	private static void fillMapOfLinksOp_B_fill_stroke() {
		links = new ArrayList<String>();
		links.add("fillCS");
		links.add("strokeCS");
		mapOfLinks.put("Op_B_fill_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_B_fill_stroke();
		fillMapOfPropertiesOp_B_fill_stroke();
		fillMapOfLinksOp_B_fill_stroke();
	}


	private static void fillMapOfSuperNamesOp_BStar_eofill_stroke() {
		mapOfSuperNames.put("Op_BStar_eofill_stroke","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_BStar_eofill_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_BStar_eofill_stroke",properties);
	}

	private static void fillMapOfLinksOp_BStar_eofill_stroke() {
		links = new ArrayList<String>();
		links.add("fillCS");
		links.add("strokeCS");
		mapOfLinks.put("Op_BStar_eofill_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_BStar_eofill_stroke();
		fillMapOfPropertiesOp_BStar_eofill_stroke();
		fillMapOfLinksOp_BStar_eofill_stroke();
	}


	private static void fillMapOfSuperNamesOp_b_closepath_fill_stroke() {
		mapOfSuperNames.put("Op_b_closepath_fill_stroke","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_b_closepath_fill_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_b_closepath_fill_stroke",properties);
	}

	private static void fillMapOfLinksOp_b_closepath_fill_stroke() {
		links = new ArrayList<String>();
		links.add("fillCS");
		links.add("strokeCS");
		mapOfLinks.put("Op_b_closepath_fill_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_b_closepath_fill_stroke();
		fillMapOfPropertiesOp_b_closepath_fill_stroke();
		fillMapOfLinksOp_b_closepath_fill_stroke();
	}


	private static void fillMapOfSuperNamesOp_bstar_closepath_eofill_stroke() {
		mapOfSuperNames.put("Op_bstar_closepath_eofill_stroke","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_bstar_closepath_eofill_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_bstar_closepath_eofill_stroke",properties);
	}

	private static void fillMapOfLinksOp_bstar_closepath_eofill_stroke() {
		links = new ArrayList<String>();
		links.add("fillCS");
		links.add("strokeCS");
		mapOfLinks.put("Op_bstar_closepath_eofill_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_bstar_closepath_eofill_stroke();
		fillMapOfPropertiesOp_bstar_closepath_eofill_stroke();
		fillMapOfLinksOp_bstar_closepath_eofill_stroke();
	}


	private static void fillMapOfSuperNamesOp_n() {
		mapOfSuperNames.put("Op_n","OpPathPaint");
	}

	private static void fillMapOfPropertiesOp_n() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_n",properties);
	}

	private static void fillMapOfLinksOp_n() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_n",links);
	}

	static {
		fillMapOfSuperNamesOp_n();
		fillMapOfPropertiesOp_n();
		fillMapOfLinksOp_n();
	}


	private static void fillMapOfSuperNamesOpClip() {
		mapOfSuperNames.put("OpClip","Operator");
	}

	private static void fillMapOfPropertiesOpClip() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpClip",properties);
	}

	private static void fillMapOfLinksOpClip() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpClip",links);
	}

	static {
		fillMapOfSuperNamesOpClip();
		fillMapOfPropertiesOpClip();
		fillMapOfLinksOpClip();
	}


	private static void fillMapOfSuperNamesOp_W_clip() {
		mapOfSuperNames.put("Op_W_clip","OpClip");
	}

	private static void fillMapOfPropertiesOp_W_clip() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_W_clip",properties);
	}

	private static void fillMapOfLinksOp_W_clip() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_W_clip",links);
	}

	static {
		fillMapOfSuperNamesOp_W_clip();
		fillMapOfPropertiesOp_W_clip();
		fillMapOfLinksOp_W_clip();
	}


	private static void fillMapOfSuperNamesOp_WStar() {
		mapOfSuperNames.put("Op_WStar","OpClip");
	}

	private static void fillMapOfPropertiesOp_WStar() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_WStar",properties);
	}

	private static void fillMapOfLinksOp_WStar() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_WStar",links);
	}

	static {
		fillMapOfSuperNamesOp_WStar();
		fillMapOfPropertiesOp_WStar();
		fillMapOfLinksOp_WStar();
	}


	private static void fillMapOfSuperNamesOpTextObject() {
		mapOfSuperNames.put("OpTextObject","Operator");
	}

	private static void fillMapOfPropertiesOpTextObject() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpTextObject",properties);
	}

	private static void fillMapOfLinksOpTextObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpTextObject",links);
	}

	static {
		fillMapOfSuperNamesOpTextObject();
		fillMapOfPropertiesOpTextObject();
		fillMapOfLinksOpTextObject();
	}


	private static void fillMapOfSuperNamesOp_ET() {
		mapOfSuperNames.put("Op_ET","OpTextObject");
	}

	private static void fillMapOfPropertiesOp_ET() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_ET",properties);
	}

	private static void fillMapOfLinksOp_ET() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_ET",links);
	}

	static {
		fillMapOfSuperNamesOp_ET();
		fillMapOfPropertiesOp_ET();
		fillMapOfLinksOp_ET();
	}


	private static void fillMapOfSuperNamesOp_BT() {
		mapOfSuperNames.put("Op_BT","OpTextObject");
	}

	private static void fillMapOfPropertiesOp_BT() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_BT",properties);
	}

	private static void fillMapOfLinksOp_BT() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_BT",links);
	}

	static {
		fillMapOfSuperNamesOp_BT();
		fillMapOfPropertiesOp_BT();
		fillMapOfLinksOp_BT();
	}


	private static void fillMapOfSuperNamesOpTextState() {
		mapOfSuperNames.put("OpTextState","Operator");
	}

	private static void fillMapOfPropertiesOpTextState() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpTextState",properties);
	}

	private static void fillMapOfLinksOpTextState() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpTextState",links);
	}

	static {
		fillMapOfSuperNamesOpTextState();
		fillMapOfPropertiesOpTextState();
		fillMapOfLinksOpTextState();
	}


	private static void fillMapOfSuperNamesOp_Tz() {
		mapOfSuperNames.put("Op_Tz","OpTextState");
	}

	private static void fillMapOfPropertiesOp_Tz() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Tz",properties);
	}

	private static void fillMapOfLinksOp_Tz() {
		links = new ArrayList<String>();
		links.add("scale");
		mapOfLinks.put("Op_Tz",links);
	}

	static {
		fillMapOfSuperNamesOp_Tz();
		fillMapOfPropertiesOp_Tz();
		fillMapOfLinksOp_Tz();
	}


	private static void fillMapOfSuperNamesOp_Tr() {
		mapOfSuperNames.put("Op_Tr","OpTextState");
	}

	private static void fillMapOfPropertiesOp_Tr() {
		properties = new ArrayList<String>();
		properties.add("renderingMode");
		mapOfProperties.put("Op_Tr",properties);
	}

	private static void fillMapOfLinksOp_Tr() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_Tr",links);
	}

	static {
		fillMapOfSuperNamesOp_Tr();
		fillMapOfPropertiesOp_Tr();
		fillMapOfLinksOp_Tr();
	}


	private static void fillMapOfSuperNamesOp_Tc() {
		mapOfSuperNames.put("Op_Tc","OpTextState");
	}

	private static void fillMapOfPropertiesOp_Tc() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Tc",properties);
	}

	private static void fillMapOfLinksOp_Tc() {
		links = new ArrayList<String>();
		links.add("charSpace");
		mapOfLinks.put("Op_Tc",links);
	}

	static {
		fillMapOfSuperNamesOp_Tc();
		fillMapOfPropertiesOp_Tc();
		fillMapOfLinksOp_Tc();
	}


	private static void fillMapOfSuperNamesOp_Tf() {
		mapOfSuperNames.put("Op_Tf","OpTextState");
	}

	private static void fillMapOfPropertiesOp_Tf() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Tf",properties);
	}

	private static void fillMapOfLinksOp_Tf() {
		links = new ArrayList<String>();
		links.add("size");
		links.add("fontName");
		mapOfLinks.put("Op_Tf",links);
	}

	static {
		fillMapOfSuperNamesOp_Tf();
		fillMapOfPropertiesOp_Tf();
		fillMapOfLinksOp_Tf();
	}


	private static void fillMapOfSuperNamesOp_Tl() {
		mapOfSuperNames.put("Op_Tl","OpTextState");
	}

	private static void fillMapOfPropertiesOp_Tl() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Tl",properties);
	}

	private static void fillMapOfLinksOp_Tl() {
		links = new ArrayList<String>();
		links.add("leading");
		mapOfLinks.put("Op_Tl",links);
	}

	static {
		fillMapOfSuperNamesOp_Tl();
		fillMapOfPropertiesOp_Tl();
		fillMapOfLinksOp_Tl();
	}


	private static void fillMapOfSuperNamesOp_Ts() {
		mapOfSuperNames.put("Op_Ts","OpTextState");
	}

	private static void fillMapOfPropertiesOp_Ts() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Ts",properties);
	}

	private static void fillMapOfLinksOp_Ts() {
		links = new ArrayList<String>();
		links.add("rise");
		mapOfLinks.put("Op_Ts",links);
	}

	static {
		fillMapOfSuperNamesOp_Ts();
		fillMapOfPropertiesOp_Ts();
		fillMapOfLinksOp_Ts();
	}


	private static void fillMapOfSuperNamesOp_Tw() {
		mapOfSuperNames.put("Op_Tw","OpTextState");
	}

	private static void fillMapOfPropertiesOp_Tw() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Tw",properties);
	}

	private static void fillMapOfLinksOp_Tw() {
		links = new ArrayList<String>();
		links.add("wordSpace");
		mapOfLinks.put("Op_Tw",links);
	}

	static {
		fillMapOfSuperNamesOp_Tw();
		fillMapOfPropertiesOp_Tw();
		fillMapOfLinksOp_Tw();
	}


	private static void fillMapOfSuperNamesOpTextPosition() {
		mapOfSuperNames.put("OpTextPosition","Operator");
	}

	private static void fillMapOfPropertiesOpTextPosition() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpTextPosition",properties);
	}

	private static void fillMapOfLinksOpTextPosition() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpTextPosition",links);
	}

	static {
		fillMapOfSuperNamesOpTextPosition();
		fillMapOfPropertiesOpTextPosition();
		fillMapOfLinksOpTextPosition();
	}


	private static void fillMapOfSuperNamesOp_Td() {
		mapOfSuperNames.put("Op_Td","OpTextPosition");
	}

	private static void fillMapOfPropertiesOp_Td() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Td",properties);
	}

	private static void fillMapOfLinksOp_Td() {
		links = new ArrayList<String>();
		links.add("horizontalOffset");
		links.add("verticalOffset");
		mapOfLinks.put("Op_Td",links);
	}

	static {
		fillMapOfSuperNamesOp_Td();
		fillMapOfPropertiesOp_Td();
		fillMapOfLinksOp_Td();
	}


	private static void fillMapOfSuperNamesOp_TD_Big() {
		mapOfSuperNames.put("Op_TD_Big","OpTextPosition");
	}

	private static void fillMapOfPropertiesOp_TD_Big() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_TD_Big",properties);
	}

	private static void fillMapOfLinksOp_TD_Big() {
		links = new ArrayList<String>();
		links.add("horizontalOffset");
		links.add("verticalOffset");
		mapOfLinks.put("Op_TD_Big",links);
	}

	static {
		fillMapOfSuperNamesOp_TD_Big();
		fillMapOfPropertiesOp_TD_Big();
		fillMapOfLinksOp_TD_Big();
	}


	private static void fillMapOfSuperNamesOp_Tm() {
		mapOfSuperNames.put("Op_Tm","OpTextPosition");
	}

	private static void fillMapOfPropertiesOp_Tm() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Tm",properties);
	}

	private static void fillMapOfLinksOp_Tm() {
		links = new ArrayList<String>();
		links.add("controlPoints");
		mapOfLinks.put("Op_Tm",links);
	}

	static {
		fillMapOfSuperNamesOp_Tm();
		fillMapOfPropertiesOp_Tm();
		fillMapOfLinksOp_Tm();
	}


	private static void fillMapOfSuperNamesOp_T_Star() {
		mapOfSuperNames.put("Op_T_Star","OpTextPosition");
	}

	private static void fillMapOfPropertiesOp_T_Star() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_T_Star",properties);
	}

	private static void fillMapOfLinksOp_T_Star() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_T_Star",links);
	}

	static {
		fillMapOfSuperNamesOp_T_Star();
		fillMapOfPropertiesOp_T_Star();
		fillMapOfLinksOp_T_Star();
	}


	private static void fillMapOfSuperNamesOpTextShow() {
		mapOfSuperNames.put("OpTextShow","Operator");
	}

	private static void fillMapOfPropertiesOpTextShow() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpTextShow",properties);
	}

	private static void fillMapOfLinksOpTextShow() {
		links = new ArrayList<String>();
		links.add("font");
		links.add("usedGlyphs");
		links.add("fillCS");
		links.add("strokeCS");
		mapOfLinks.put("OpTextShow",links);
	}

	static {
		fillMapOfSuperNamesOpTextShow();
		fillMapOfPropertiesOpTextShow();
		fillMapOfLinksOpTextShow();
	}


	private static void fillMapOfSuperNamesOp_Tj() {
		mapOfSuperNames.put("Op_Tj","OpTextShow");
	}

	private static void fillMapOfPropertiesOp_Tj() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Tj",properties);
	}

	private static void fillMapOfLinksOp_Tj() {
		links = new ArrayList<String>();
		links.add("showString");
		mapOfLinks.put("Op_Tj",links);
	}

	static {
		fillMapOfSuperNamesOp_Tj();
		fillMapOfPropertiesOp_Tj();
		fillMapOfLinksOp_Tj();
	}


	private static void fillMapOfSuperNamesOp_TJ_Big() {
		mapOfSuperNames.put("Op_TJ_Big","OpTextShow");
	}

	private static void fillMapOfPropertiesOp_TJ_Big() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_TJ_Big",properties);
	}

	private static void fillMapOfLinksOp_TJ_Big() {
		links = new ArrayList<String>();
		links.add("specialStrings");
		mapOfLinks.put("Op_TJ_Big",links);
	}

	static {
		fillMapOfSuperNamesOp_TJ_Big();
		fillMapOfPropertiesOp_TJ_Big();
		fillMapOfLinksOp_TJ_Big();
	}


	private static void fillMapOfSuperNamesOp_Quote() {
		mapOfSuperNames.put("Op_Quote","OpTextShow");
	}

	private static void fillMapOfPropertiesOp_Quote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Quote",properties);
	}

	private static void fillMapOfLinksOp_Quote() {
		links = new ArrayList<String>();
		links.add("showString");
		mapOfLinks.put("Op_Quote",links);
	}

	static {
		fillMapOfSuperNamesOp_Quote();
		fillMapOfPropertiesOp_Quote();
		fillMapOfLinksOp_Quote();
	}


	private static void fillMapOfSuperNamesOp_DoubleQuote() {
		mapOfSuperNames.put("Op_DoubleQuote","OpTextShow");
	}

	private static void fillMapOfPropertiesOp_DoubleQuote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_DoubleQuote",properties);
	}

	private static void fillMapOfLinksOp_DoubleQuote() {
		links = new ArrayList<String>();
		links.add("showString");
		links.add("wordSpacing");
		links.add("characterSpacing");
		mapOfLinks.put("Op_DoubleQuote",links);
	}

	static {
		fillMapOfSuperNamesOp_DoubleQuote();
		fillMapOfPropertiesOp_DoubleQuote();
		fillMapOfLinksOp_DoubleQuote();
	}


	private static void fillMapOfSuperNamesOpType3Font() {
		mapOfSuperNames.put("OpType3Font","Operator");
	}

	private static void fillMapOfPropertiesOpType3Font() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpType3Font",properties);
	}

	private static void fillMapOfLinksOpType3Font() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpType3Font",links);
	}

	static {
		fillMapOfSuperNamesOpType3Font();
		fillMapOfPropertiesOpType3Font();
		fillMapOfLinksOpType3Font();
	}


	private static void fillMapOfSuperNamesOp_d0() {
		mapOfSuperNames.put("Op_d0","OpType3Font");
	}

	private static void fillMapOfPropertiesOp_d0() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_d0",properties);
	}

	private static void fillMapOfLinksOp_d0() {
		links = new ArrayList<String>();
		links.add("horizontalDisplacement");
		links.add("verticalDisplacement");
		mapOfLinks.put("Op_d0",links);
	}

	static {
		fillMapOfSuperNamesOp_d0();
		fillMapOfPropertiesOp_d0();
		fillMapOfLinksOp_d0();
	}


	private static void fillMapOfSuperNamesOp_d1() {
		mapOfSuperNames.put("Op_d1","OpType3Font");
	}

	private static void fillMapOfPropertiesOp_d1() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_d1",properties);
	}

	private static void fillMapOfLinksOp_d1() {
		links = new ArrayList<String>();
		links.add("controlPoints");
		mapOfLinks.put("Op_d1",links);
	}

	static {
		fillMapOfSuperNamesOp_d1();
		fillMapOfPropertiesOp_d1();
		fillMapOfLinksOp_d1();
	}


	private static void fillMapOfSuperNamesOpSetColor() {
		mapOfSuperNames.put("OpSetColor","Operator");
	}

	private static void fillMapOfPropertiesOpSetColor() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpSetColor",properties);
	}

	private static void fillMapOfLinksOpSetColor() {
		links = new ArrayList<String>();
		links.add("colorValues");
		mapOfLinks.put("OpSetColor",links);
	}

	static {
		fillMapOfSuperNamesOpSetColor();
		fillMapOfPropertiesOpSetColor();
		fillMapOfLinksOpSetColor();
	}


	private static void fillMapOfSuperNamesOp_SC_stroke() {
		mapOfSuperNames.put("Op_SC_stroke","OpSetColor");
	}

	private static void fillMapOfPropertiesOp_SC_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_SC_stroke",properties);
	}

	private static void fillMapOfLinksOp_SC_stroke() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_SC_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_SC_stroke();
		fillMapOfPropertiesOp_SC_stroke();
		fillMapOfLinksOp_SC_stroke();
	}


	private static void fillMapOfSuperNamesOp_sc_fill() {
		mapOfSuperNames.put("Op_sc_fill","OpSetColor");
	}

	private static void fillMapOfPropertiesOp_sc_fill() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_sc_fill",properties);
	}

	private static void fillMapOfLinksOp_sc_fill() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_sc_fill",links);
	}

	static {
		fillMapOfSuperNamesOp_sc_fill();
		fillMapOfPropertiesOp_sc_fill();
		fillMapOfLinksOp_sc_fill();
	}


	private static void fillMapOfSuperNamesOpColor() {
		mapOfSuperNames.put("OpColor","OpSetColor");
	}

	private static void fillMapOfPropertiesOpColor() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpColor",properties);
	}

	private static void fillMapOfLinksOpColor() {
		links = new ArrayList<String>();
		links.add("colorSpace");
		links.add("patternName");
		mapOfLinks.put("OpColor",links);
	}

	static {
		fillMapOfSuperNamesOpColor();
		fillMapOfPropertiesOpColor();
		fillMapOfLinksOpColor();
	}


	private static void fillMapOfSuperNamesOp_SCN_stroke() {
		mapOfSuperNames.put("Op_SCN_stroke","OpColor");
	}

	private static void fillMapOfPropertiesOp_SCN_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_SCN_stroke",properties);
	}

	private static void fillMapOfLinksOp_SCN_stroke() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_SCN_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_SCN_stroke();
		fillMapOfPropertiesOp_SCN_stroke();
		fillMapOfLinksOp_SCN_stroke();
	}


	private static void fillMapOfSuperNamesOp_scn_fill() {
		mapOfSuperNames.put("Op_scn_fill","OpColor");
	}

	private static void fillMapOfPropertiesOp_scn_fill() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_scn_fill",properties);
	}

	private static void fillMapOfLinksOp_scn_fill() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_scn_fill",links);
	}

	static {
		fillMapOfSuperNamesOp_scn_fill();
		fillMapOfPropertiesOp_scn_fill();
		fillMapOfLinksOp_scn_fill();
	}


	private static void fillMapOfSuperNamesOp_G_stroke() {
		mapOfSuperNames.put("Op_G_stroke","OpColor");
	}

	private static void fillMapOfPropertiesOp_G_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_G_stroke",properties);
	}

	private static void fillMapOfLinksOp_G_stroke() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_G_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_G_stroke();
		fillMapOfPropertiesOp_G_stroke();
		fillMapOfLinksOp_G_stroke();
	}


	private static void fillMapOfSuperNamesOp_g_fill() {
		mapOfSuperNames.put("Op_g_fill","OpColor");
	}

	private static void fillMapOfPropertiesOp_g_fill() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_g_fill",properties);
	}

	private static void fillMapOfLinksOp_g_fill() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_g_fill",links);
	}

	static {
		fillMapOfSuperNamesOp_g_fill();
		fillMapOfPropertiesOp_g_fill();
		fillMapOfLinksOp_g_fill();
	}


	private static void fillMapOfSuperNamesOp_RG_stroke() {
		mapOfSuperNames.put("Op_RG_stroke","OpColor");
	}

	private static void fillMapOfPropertiesOp_RG_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_RG_stroke",properties);
	}

	private static void fillMapOfLinksOp_RG_stroke() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_RG_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_RG_stroke();
		fillMapOfPropertiesOp_RG_stroke();
		fillMapOfLinksOp_RG_stroke();
	}


	private static void fillMapOfSuperNamesOp_rg_fill() {
		mapOfSuperNames.put("Op_rg_fill","OpColor");
	}

	private static void fillMapOfPropertiesOp_rg_fill() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_rg_fill",properties);
	}

	private static void fillMapOfLinksOp_rg_fill() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_rg_fill",links);
	}

	static {
		fillMapOfSuperNamesOp_rg_fill();
		fillMapOfPropertiesOp_rg_fill();
		fillMapOfLinksOp_rg_fill();
	}


	private static void fillMapOfSuperNamesOp_K_stroke() {
		mapOfSuperNames.put("Op_K_stroke","OpColor");
	}

	private static void fillMapOfPropertiesOp_K_stroke() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_K_stroke",properties);
	}

	private static void fillMapOfLinksOp_K_stroke() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_K_stroke",links);
	}

	static {
		fillMapOfSuperNamesOp_K_stroke();
		fillMapOfPropertiesOp_K_stroke();
		fillMapOfLinksOp_K_stroke();
	}


	private static void fillMapOfSuperNamesOp_k_fill() {
		mapOfSuperNames.put("Op_k_fill","OpColor");
	}

	private static void fillMapOfPropertiesOp_k_fill() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_k_fill",properties);
	}

	private static void fillMapOfLinksOp_k_fill() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_k_fill",links);
	}

	static {
		fillMapOfSuperNamesOp_k_fill();
		fillMapOfPropertiesOp_k_fill();
		fillMapOfLinksOp_k_fill();
	}


	private static void fillMapOfSuperNamesOpShading() {
		mapOfSuperNames.put("OpShading","Operator");
	}

	private static void fillMapOfPropertiesOpShading() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpShading",properties);
	}

	private static void fillMapOfLinksOpShading() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpShading",links);
	}

	static {
		fillMapOfSuperNamesOpShading();
		fillMapOfPropertiesOpShading();
		fillMapOfLinksOpShading();
	}


	private static void fillMapOfSuperNamesOp_sh() {
		mapOfSuperNames.put("Op_sh","OpShading");
	}

	private static void fillMapOfPropertiesOp_sh() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_sh",properties);
	}

	private static void fillMapOfLinksOp_sh() {
		links = new ArrayList<String>();
		links.add("shading");
		mapOfLinks.put("Op_sh",links);
	}

	static {
		fillMapOfSuperNamesOp_sh();
		fillMapOfPropertiesOp_sh();
		fillMapOfLinksOp_sh();
	}


	private static void fillMapOfSuperNamesOpInlineImage() {
		mapOfSuperNames.put("OpInlineImage","Operator");
	}

	private static void fillMapOfPropertiesOpInlineImage() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpInlineImage",properties);
	}

	private static void fillMapOfLinksOpInlineImage() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpInlineImage",links);
	}

	static {
		fillMapOfSuperNamesOpInlineImage();
		fillMapOfPropertiesOpInlineImage();
		fillMapOfLinksOpInlineImage();
	}


	private static void fillMapOfSuperNamesOp_BI() {
		mapOfSuperNames.put("Op_BI","OpInlineImage");
	}

	private static void fillMapOfPropertiesOp_BI() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_BI",properties);
	}

	private static void fillMapOfLinksOp_BI() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_BI",links);
	}

	static {
		fillMapOfSuperNamesOp_BI();
		fillMapOfPropertiesOp_BI();
		fillMapOfLinksOp_BI();
	}


	private static void fillMapOfSuperNamesOp_ID() {
		mapOfSuperNames.put("Op_ID","OpInlineImage");
	}

	private static void fillMapOfPropertiesOp_ID() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_ID",properties);
	}

	private static void fillMapOfLinksOp_ID() {
		links = new ArrayList<String>();
		links.add("inlineImageDictionary");
		mapOfLinks.put("Op_ID",links);
	}

	static {
		fillMapOfSuperNamesOp_ID();
		fillMapOfPropertiesOp_ID();
		fillMapOfLinksOp_ID();
	}


	private static void fillMapOfSuperNamesOp_EI() {
		mapOfSuperNames.put("Op_EI","OpInlineImage");
	}

	private static void fillMapOfPropertiesOp_EI() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_EI",properties);
	}

	private static void fillMapOfLinksOp_EI() {
		links = new ArrayList<String>();
		links.add("inlineImage");
		mapOfLinks.put("Op_EI",links);
	}

	static {
		fillMapOfSuperNamesOp_EI();
		fillMapOfPropertiesOp_EI();
		fillMapOfLinksOp_EI();
	}


	private static void fillMapOfSuperNamesOpXObject() {
		mapOfSuperNames.put("OpXObject","Operator");
	}

	private static void fillMapOfPropertiesOpXObject() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpXObject",properties);
	}

	private static void fillMapOfLinksOpXObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpXObject",links);
	}

	static {
		fillMapOfSuperNamesOpXObject();
		fillMapOfPropertiesOpXObject();
		fillMapOfLinksOpXObject();
	}


	private static void fillMapOfSuperNamesOp_Do() {
		mapOfSuperNames.put("Op_Do","OpXObject");
	}

	private static void fillMapOfPropertiesOp_Do() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_Do",properties);
	}

	private static void fillMapOfLinksOp_Do() {
		links = new ArrayList<String>();
		links.add("xObject");
		mapOfLinks.put("Op_Do",links);
	}

	static {
		fillMapOfSuperNamesOp_Do();
		fillMapOfPropertiesOp_Do();
		fillMapOfLinksOp_Do();
	}


	private static void fillMapOfSuperNamesOpMarkedContent() {
		mapOfSuperNames.put("OpMarkedContent","Operator");
	}

	private static void fillMapOfPropertiesOpMarkedContent() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpMarkedContent",properties);
	}

	private static void fillMapOfLinksOpMarkedContent() {
		links = new ArrayList<String>();
		links.add("actualText");
		mapOfLinks.put("OpMarkedContent",links);
	}

	static {
		fillMapOfSuperNamesOpMarkedContent();
		fillMapOfPropertiesOpMarkedContent();
		fillMapOfLinksOpMarkedContent();
	}


	private static void fillMapOfSuperNamesOp_MP() {
		mapOfSuperNames.put("Op_MP","OpMarkedContent");
	}

	private static void fillMapOfPropertiesOp_MP() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_MP",properties);
	}

	private static void fillMapOfLinksOp_MP() {
		links = new ArrayList<String>();
		links.add("tag");
		mapOfLinks.put("Op_MP",links);
	}

	static {
		fillMapOfSuperNamesOp_MP();
		fillMapOfPropertiesOp_MP();
		fillMapOfLinksOp_MP();
	}


	private static void fillMapOfSuperNamesOp_DP() {
		mapOfSuperNames.put("Op_DP","OpMarkedContent");
	}

	private static void fillMapOfPropertiesOp_DP() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_DP",properties);
	}

	private static void fillMapOfLinksOp_DP() {
		links = new ArrayList<String>();
		links.add("tag");
		links.add("properties");
		mapOfLinks.put("Op_DP",links);
	}

	static {
		fillMapOfSuperNamesOp_DP();
		fillMapOfPropertiesOp_DP();
		fillMapOfLinksOp_DP();
	}


	private static void fillMapOfSuperNamesOp_BMC() {
		mapOfSuperNames.put("Op_BMC","OpMarkedContent");
	}

	private static void fillMapOfPropertiesOp_BMC() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_BMC",properties);
	}

	private static void fillMapOfLinksOp_BMC() {
		links = new ArrayList<String>();
		links.add("tag");
		mapOfLinks.put("Op_BMC",links);
	}

	static {
		fillMapOfSuperNamesOp_BMC();
		fillMapOfPropertiesOp_BMC();
		fillMapOfLinksOp_BMC();
	}


	private static void fillMapOfSuperNamesOp_BDC() {
		mapOfSuperNames.put("Op_BDC","OpMarkedContent");
	}

	private static void fillMapOfPropertiesOp_BDC() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_BDC",properties);
	}

	private static void fillMapOfLinksOp_BDC() {
		links = new ArrayList<String>();
		links.add("tag");
		links.add("properties");
		links.add("Lang");
		mapOfLinks.put("Op_BDC",links);
	}

	static {
		fillMapOfSuperNamesOp_BDC();
		fillMapOfPropertiesOp_BDC();
		fillMapOfLinksOp_BDC();
	}


	private static void fillMapOfSuperNamesOp_EMC() {
		mapOfSuperNames.put("Op_EMC","OpMarkedContent");
	}

	private static void fillMapOfPropertiesOp_EMC() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_EMC",properties);
	}

	private static void fillMapOfLinksOp_EMC() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_EMC",links);
	}

	static {
		fillMapOfSuperNamesOp_EMC();
		fillMapOfPropertiesOp_EMC();
		fillMapOfLinksOp_EMC();
	}


	private static void fillMapOfSuperNamesOpCompatibility() {
		mapOfSuperNames.put("OpCompatibility","Operator");
	}

	private static void fillMapOfPropertiesOpCompatibility() {
		properties = new ArrayList<String>();
		mapOfProperties.put("OpCompatibility",properties);
	}

	private static void fillMapOfLinksOpCompatibility() {
		links = new ArrayList<String>();
		mapOfLinks.put("OpCompatibility",links);
	}

	static {
		fillMapOfSuperNamesOpCompatibility();
		fillMapOfPropertiesOpCompatibility();
		fillMapOfLinksOpCompatibility();
	}


	private static void fillMapOfSuperNamesOp_BX() {
		mapOfSuperNames.put("Op_BX","OpCompatibility");
	}

	private static void fillMapOfPropertiesOp_BX() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_BX",properties);
	}

	private static void fillMapOfLinksOp_BX() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_BX",links);
	}

	static {
		fillMapOfSuperNamesOp_BX();
		fillMapOfPropertiesOp_BX();
		fillMapOfLinksOp_BX();
	}


	private static void fillMapOfSuperNamesOp_EX() {
		mapOfSuperNames.put("Op_EX","OpCompatibility");
	}

	private static void fillMapOfPropertiesOp_EX() {
		properties = new ArrayList<String>();
		mapOfProperties.put("Op_EX",properties);
	}

	private static void fillMapOfLinksOp_EX() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_EX",links);
	}

	static {
		fillMapOfSuperNamesOp_EX();
		fillMapOfPropertiesOp_EX();
		fillMapOfLinksOp_EX();
	}


	private static void fillMapOfSuperNamesOp_Undefined() {
		mapOfSuperNames.put("Op_Undefined","OpCompatibility");
	}

	private static void fillMapOfPropertiesOp_Undefined() {
		properties = new ArrayList<String>();
		properties.add("name");
		mapOfProperties.put("Op_Undefined",properties);
	}

	private static void fillMapOfLinksOp_Undefined() {
		links = new ArrayList<String>();
		mapOfLinks.put("Op_Undefined",links);
	}

	static {
		fillMapOfSuperNamesOp_Undefined();
		fillMapOfPropertiesOp_Undefined();
		fillMapOfLinksOp_Undefined();
	}


	private static void fillMapOfSuperNamesGlyph() {
		mapOfSuperNames.put("Glyph","Object");
	}

	private static void fillMapOfPropertiesGlyph() {
		properties = new ArrayList<String>();
		properties.add("name");
		properties.add("widthFromDictionary");
		properties.add("widthFromFontProgram");
		properties.add("isGlyphPresent");
		properties.add("toUnicode");
		properties.add("renderingMode");
		properties.add("unicodePUA");
		properties.add("actualTextPresent");
		mapOfProperties.put("Glyph",properties);
	}

	private static void fillMapOfLinksGlyph() {
		links = new ArrayList<String>();
		mapOfLinks.put("Glyph",links);
	}

	static {
		fillMapOfSuperNamesGlyph();
		fillMapOfPropertiesGlyph();
		fillMapOfLinksGlyph();
	}


	private static void fillMapOfSuperNamesCIDGlyph() {
		mapOfSuperNames.put("CIDGlyph","Glyph");
	}

	private static void fillMapOfPropertiesCIDGlyph() {
		properties = new ArrayList<String>();
		properties.add("CID");
		mapOfProperties.put("CIDGlyph",properties);
	}

	private static void fillMapOfLinksCIDGlyph() {
		links = new ArrayList<String>();
		mapOfLinks.put("CIDGlyph",links);
	}

	static {
		fillMapOfSuperNamesCIDGlyph();
		fillMapOfPropertiesCIDGlyph();
		fillMapOfLinksCIDGlyph();
		}


	private static void fillMapOfSuperNamesSENonStandard() {
		mapOfSuperNames.put("SENonStandard","PDStructElem");
	}

	private static void fillMapOfPropertiesSENonStandard() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SENonStandard",properties);
	}

	private static void fillMapOfLinksSENonStandard() {
		links = new ArrayList<String>();
		mapOfLinks.put("SENonStandard",links);
	}

	static {
		fillMapOfSuperNamesSENonStandard();
		fillMapOfPropertiesSENonStandard();
		fillMapOfLinksSENonStandard();
	}


	private static void fillMapOfSuperNamesSEDocument() {
		mapOfSuperNames.put("SEDocument","PDStructElem");
	}

	private static void fillMapOfPropertiesSEDocument() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEDocument",properties);
	}

	private static void fillMapOfLinksSEDocument() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEDocument",links);
	}

	static {
		fillMapOfSuperNamesSEDocument();
		fillMapOfPropertiesSEDocument();
		fillMapOfLinksSEDocument();
	}


	private static void fillMapOfSuperNamesSEPart() {
		mapOfSuperNames.put("SEPart","PDStructElem");
	}

	private static void fillMapOfPropertiesSEPart() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEPart",properties);
	}

	private static void fillMapOfLinksSEPart() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEPart",links);
	}

	static {
		fillMapOfSuperNamesSEPart();
		fillMapOfPropertiesSEPart();
		fillMapOfLinksSEPart();
	}


	private static void fillMapOfSuperNamesSEDiv() {
		mapOfSuperNames.put("SEDiv","PDStructElem");
	}

	private static void fillMapOfPropertiesSEDiv() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEDiv",properties);
	}

	private static void fillMapOfLinksSEDiv() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEDiv",links);
	}

	static {
		fillMapOfSuperNamesSEDiv();
		fillMapOfPropertiesSEDiv();
		fillMapOfLinksSEDiv();
	}


	private static void fillMapOfSuperNamesSECaption() {
		mapOfSuperNames.put("SECaption","PDStructElem");
	}

	private static void fillMapOfPropertiesSECaption() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SECaption",properties);
	}

	private static void fillMapOfLinksSECaption() {
		links = new ArrayList<String>();
		mapOfLinks.put("SECaption",links);
	}

	static {
		fillMapOfSuperNamesSECaption();
		fillMapOfPropertiesSECaption();
		fillMapOfLinksSECaption();
	}


	private static void fillMapOfSuperNamesSEH() {
		mapOfSuperNames.put("SEH","PDStructElem");
	}

	private static void fillMapOfPropertiesSEH() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEH",properties);
	}

	private static void fillMapOfLinksSEH() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEH",links);
	}

	static {
		fillMapOfSuperNamesSEH();
		fillMapOfPropertiesSEH();
		fillMapOfLinksSEH();
	}


	private static void fillMapOfSuperNamesSEP() {
		mapOfSuperNames.put("SEP","PDStructElem");
	}

	private static void fillMapOfPropertiesSEP() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEP",properties);
	}

	private static void fillMapOfLinksSEP() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEP",links);
	}

	static {
		fillMapOfSuperNamesSEP();
		fillMapOfPropertiesSEP();
		fillMapOfLinksSEP();
	}


	private static void fillMapOfSuperNamesSEL() {
		mapOfSuperNames.put("SEL","PDStructElem");
	}

	private static void fillMapOfPropertiesSEL() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEL",properties);
	}

	private static void fillMapOfLinksSEL() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEL",links);
	}

	static {
		fillMapOfSuperNamesSEL();
		fillMapOfPropertiesSEL();
		fillMapOfLinksSEL();
	}


	private static void fillMapOfSuperNamesSELI() {
		mapOfSuperNames.put("SELI","PDStructElem");
	}

	private static void fillMapOfPropertiesSELI() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SELI",properties);
	}

	private static void fillMapOfLinksSELI() {
		links = new ArrayList<String>();
		mapOfLinks.put("SELI",links);
	}

	static {
		fillMapOfSuperNamesSELI();
		fillMapOfPropertiesSELI();
		fillMapOfLinksSELI();
	}


	private static void fillMapOfSuperNamesSELbl() {
		mapOfSuperNames.put("SELbl","PDStructElem");
	}

	private static void fillMapOfPropertiesSELbl() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SELbl",properties);
	}

	private static void fillMapOfLinksSELbl() {
		links = new ArrayList<String>();
		mapOfLinks.put("SELbl",links);
	}

	static {
		fillMapOfSuperNamesSELbl();
		fillMapOfPropertiesSELbl();
		fillMapOfLinksSELbl();
	}


	private static void fillMapOfSuperNamesSELBody() {
		mapOfSuperNames.put("SELBody","PDStructElem");
	}

	private static void fillMapOfPropertiesSELBody() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SELBody",properties);
	}

	private static void fillMapOfLinksSELBody() {
		links = new ArrayList<String>();
		mapOfLinks.put("SELBody",links);
	}

	static {
		fillMapOfSuperNamesSELBody();
		fillMapOfPropertiesSELBody();
		fillMapOfLinksSELBody();
	}


	private static void fillMapOfSuperNamesSETable() {
		mapOfSuperNames.put("SETable","PDStructElem");
	}

	private static void fillMapOfPropertiesSETable() {
		properties = new ArrayList<String>();
		properties.add("useHeadersAndIdOrScope");
		properties.add("columnSpan");
		properties.add("rowSpan");
		properties.add("numberOfRowWithWrongColumnSpan");
		properties.add("numberOfColumnWithWrongRowSpan");
		properties.add("wrongColumnSpan");
		mapOfProperties.put("SETable",properties);
	}

	private static void fillMapOfLinksSETable() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETable",links);
	}

	static {
		fillMapOfSuperNamesSETable();
		fillMapOfPropertiesSETable();
		fillMapOfLinksSETable();
	}


	private static void fillMapOfSuperNamesSETR() {
		mapOfSuperNames.put("SETR","PDStructElem");
	}

	private static void fillMapOfPropertiesSETR() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SETR",properties);
	}

	private static void fillMapOfLinksSETR() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETR",links);
	}

	static {
		fillMapOfSuperNamesSETR();
		fillMapOfPropertiesSETR();
		fillMapOfLinksSETR();
	}


	private static void fillMapOfSuperNamesSETableCell() {
		mapOfSuperNames.put("SETableCell","PDStructElem");
	}

	private static void fillMapOfPropertiesSETableCell() {
		properties = new ArrayList<String>();
		properties.add("ColSpan");
		properties.add("RowSpan");
		properties.add("hasIntersection");
		mapOfProperties.put("SETableCell",properties);
	}

	private static void fillMapOfLinksSETableCell() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETableCell",links);
	}

	static {
		fillMapOfSuperNamesSETableCell();
		fillMapOfPropertiesSETableCell();
		fillMapOfLinksSETableCell();
	}


	private static void fillMapOfSuperNamesSETH() {
		mapOfSuperNames.put("SETH","SETableCell");
	}

	private static void fillMapOfPropertiesSETH() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SETH",properties);
	}

	private static void fillMapOfLinksSETH() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETH",links);
	}

	static {
		fillMapOfSuperNamesSETH();
		fillMapOfPropertiesSETH();
		fillMapOfLinksSETH();
	}


	private static void fillMapOfSuperNamesSETD() {
		mapOfSuperNames.put("SETD","SETableCell");
	}

	private static void fillMapOfPropertiesSETD() {
		properties = new ArrayList<String>();
		properties.add("hasConnectedHeader");
		properties.add("unknownHeaders");
		mapOfProperties.put("SETD",properties);
	}

	private static void fillMapOfLinksSETD() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETD",links);
	}

	static {
		fillMapOfSuperNamesSETD();
		fillMapOfPropertiesSETD();
		fillMapOfLinksSETD();
	}


	private static void fillMapOfSuperNamesSESpan() {
		mapOfSuperNames.put("SESpan","PDStructElem");
	}

	private static void fillMapOfPropertiesSESpan() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SESpan",properties);
	}

	private static void fillMapOfLinksSESpan() {
		links = new ArrayList<String>();
		mapOfLinks.put("SESpan",links);
	}

	static {
		fillMapOfSuperNamesSESpan();
		fillMapOfPropertiesSESpan();
		fillMapOfLinksSESpan();
	}


	private static void fillMapOfSuperNamesSELink() {
		mapOfSuperNames.put("SELink","PDStructElem");
	}

	private static void fillMapOfPropertiesSELink() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SELink",properties);
	}

	private static void fillMapOfLinksSELink() {
		links = new ArrayList<String>();
		mapOfLinks.put("SELink",links);
	}

	static {
		fillMapOfSuperNamesSELink();
		fillMapOfPropertiesSELink();
		fillMapOfLinksSELink();
	}


	private static void fillMapOfSuperNamesSEArt() {
		mapOfSuperNames.put("SEArt","PDStructElem");
	}

	private static void fillMapOfPropertiesSEArt() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEArt",properties);
	}

	private static void fillMapOfLinksSEArt() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEArt",links);
	}

	static {
		fillMapOfSuperNamesSEArt();
		fillMapOfPropertiesSEArt();
		fillMapOfLinksSEArt();
	}


	private static void fillMapOfSuperNamesSESect() {
		mapOfSuperNames.put("SESect","PDStructElem");
	}

	private static void fillMapOfPropertiesSESect() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SESect",properties);
	}

	private static void fillMapOfLinksSESect() {
		links = new ArrayList<String>();
		mapOfLinks.put("SESect",links);
	}

	static {
		fillMapOfSuperNamesSESect();
		fillMapOfPropertiesSESect();
		fillMapOfLinksSESect();
	}


	private static void fillMapOfSuperNamesSEBlockQuote() {
		mapOfSuperNames.put("SEBlockQuote","PDStructElem");
	}

	private static void fillMapOfPropertiesSEBlockQuote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEBlockQuote",properties);
	}

	private static void fillMapOfLinksSEBlockQuote() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEBlockQuote",links);
	}

	static {
		fillMapOfSuperNamesSEBlockQuote();
		fillMapOfPropertiesSEBlockQuote();
		fillMapOfLinksSEBlockQuote();
	}


	private static void fillMapOfSuperNamesSETOC() {
		mapOfSuperNames.put("SETOC","PDStructElem");
	}

	private static void fillMapOfPropertiesSETOC() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SETOC",properties);
	}

	private static void fillMapOfLinksSETOC() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETOC",links);
	}

	static {
		fillMapOfSuperNamesSETOC();
		fillMapOfPropertiesSETOC();
		fillMapOfLinksSETOC();
	}


	private static void fillMapOfSuperNamesSETOCI() {
		mapOfSuperNames.put("SETOCI","PDStructElem");
	}

	private static void fillMapOfPropertiesSETOCI() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SETOCI",properties);
	}

	private static void fillMapOfLinksSETOCI() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETOCI",links);
	}

	static {
		fillMapOfSuperNamesSETOCI();
		fillMapOfPropertiesSETOCI();
		fillMapOfLinksSETOCI();
	}


	private static void fillMapOfSuperNamesSEIndex() {
		mapOfSuperNames.put("SEIndex","PDStructElem");
	}

	private static void fillMapOfPropertiesSEIndex() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEIndex",properties);
	}

	private static void fillMapOfLinksSEIndex() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEIndex",links);
	}

	static {
		fillMapOfSuperNamesSEIndex();
		fillMapOfPropertiesSEIndex();
		fillMapOfLinksSEIndex();
	}


	private static void fillMapOfSuperNamesSENonStruct() {
		mapOfSuperNames.put("SENonStruct","PDStructElem");
	}

	private static void fillMapOfPropertiesSENonStruct() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SENonStruct",properties);
	}

	private static void fillMapOfLinksSENonStruct() {
		links = new ArrayList<String>();
		mapOfLinks.put("SENonStruct",links);
	}

	static {
		fillMapOfSuperNamesSENonStruct();
		fillMapOfPropertiesSENonStruct();
		fillMapOfLinksSENonStruct();
	}


	private static void fillMapOfSuperNamesSEPrivate() {
		mapOfSuperNames.put("SEPrivate","PDStructElem");
	}

	private static void fillMapOfPropertiesSEPrivate() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEPrivate",properties);
	}

	private static void fillMapOfLinksSEPrivate() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEPrivate",links);
	}

	static {
		fillMapOfSuperNamesSEPrivate();
		fillMapOfPropertiesSEPrivate();
		fillMapOfLinksSEPrivate();
	}


	private static void fillMapOfSuperNamesSEQuote() {
		mapOfSuperNames.put("SEQuote","PDStructElem");
	}

	private static void fillMapOfPropertiesSEQuote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEQuote",properties);
	}

	private static void fillMapOfLinksSEQuote() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEQuote",links);
	}

	static {
		fillMapOfSuperNamesSEQuote();
		fillMapOfPropertiesSEQuote();
		fillMapOfLinksSEQuote();
	}


	private static void fillMapOfSuperNamesSENote() {
		mapOfSuperNames.put("SENote","PDStructElem");
	}

	private static void fillMapOfPropertiesSENote() {
		properties = new ArrayList<String>();
		properties.add("noteID");
		properties.add("hasDuplicateNoteID");
		mapOfProperties.put("SENote",properties);
	}

	private static void fillMapOfLinksSENote() {
		links = new ArrayList<String>();
		mapOfLinks.put("SENote",links);
	}

	static {
		fillMapOfSuperNamesSENote();
		fillMapOfPropertiesSENote();
		fillMapOfLinksSENote();
	}


	private static void fillMapOfSuperNamesSEReference() {
		mapOfSuperNames.put("SEReference","PDStructElem");
	}

	private static void fillMapOfPropertiesSEReference() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEReference",properties);
	}

	private static void fillMapOfLinksSEReference() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEReference",links);
	}

	static {
		fillMapOfSuperNamesSEReference();
		fillMapOfPropertiesSEReference();
		fillMapOfLinksSEReference();
	}


	private static void fillMapOfSuperNamesSEBibEntry() {
		mapOfSuperNames.put("SEBibEntry","PDStructElem");
	}

	private static void fillMapOfPropertiesSEBibEntry() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEBibEntry",properties);
	}

	private static void fillMapOfLinksSEBibEntry() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEBibEntry",links);
	}

	static {
		fillMapOfSuperNamesSEBibEntry();
		fillMapOfPropertiesSEBibEntry();
		fillMapOfLinksSEBibEntry();
	}


	private static void fillMapOfSuperNamesSECode() {
		mapOfSuperNames.put("SECode","PDStructElem");
	}

	private static void fillMapOfPropertiesSECode() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SECode",properties);
	}

	private static void fillMapOfLinksSECode() {
		links = new ArrayList<String>();
		mapOfLinks.put("SECode",links);
	}

	static {
		fillMapOfSuperNamesSECode();
		fillMapOfPropertiesSECode();
		fillMapOfLinksSECode();
	}


	private static void fillMapOfSuperNamesSEHn() {
		mapOfSuperNames.put("SEHn","PDStructElem");
	}

	private static void fillMapOfPropertiesSEHn() {
		properties = new ArrayList<String>();
		properties.add("hasCorrectNestingLevel");
		properties.add("nestingLevel");
		mapOfProperties.put("SEHn",properties);
	}

	private static void fillMapOfLinksSEHn() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEHn",links);
	}

	static {
		fillMapOfSuperNamesSEHn();
		fillMapOfPropertiesSEHn();
		fillMapOfLinksSEHn();
	}


	private static void fillMapOfSuperNamesSEFigure() {
		mapOfSuperNames.put("SEFigure","PDStructElem");
	}

	private static void fillMapOfPropertiesSEFigure() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEFigure",properties);
	}

	private static void fillMapOfLinksSEFigure() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEFigure",links);
	}

	static {
		fillMapOfSuperNamesSEFigure();
		fillMapOfPropertiesSEFigure();
		fillMapOfLinksSEFigure();
	}


	private static void fillMapOfSuperNamesSEFormula() {
		mapOfSuperNames.put("SEFormula","PDStructElem");
	}

	private static void fillMapOfPropertiesSEFormula() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEFormula",properties);
	}

	private static void fillMapOfLinksSEFormula() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEFormula",links);
	}

	static {
		fillMapOfSuperNamesSEFormula();
		fillMapOfPropertiesSEFormula();
		fillMapOfLinksSEFormula();
	}


	private static void fillMapOfSuperNamesSEContentItem() {
		mapOfSuperNames.put("SEContentItem","Object");
	}

	private static void fillMapOfPropertiesSEContentItem() {
		properties = new ArrayList<String>();
		properties.add("parentsTags");
		properties.add("parentStructureTag");
		properties.add("parentStandardTag");
		mapOfProperties.put("SEContentItem",properties);
	}

	private static void fillMapOfLinksSEContentItem() {
		links = new ArrayList<String>();
		links.add("contentItem");
		mapOfLinks.put("SEContentItem",links);
	}

	static {
		fillMapOfSuperNamesSEContentItem();
		fillMapOfPropertiesSEContentItem();
		fillMapOfLinksSEContentItem();
	}


	private static void fillMapOfSuperNamesSESimpleContentItem() {
		mapOfSuperNames.put("SESimpleContentItem","SEContentItem");
	}

	private static void fillMapOfPropertiesSESimpleContentItem() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SESimpleContentItem",properties);
	}

	private static void fillMapOfLinksSESimpleContentItem() {
		links = new ArrayList<String>();
		mapOfLinks.put("SESimpleContentItem",links);
	}

	static {
		fillMapOfSuperNamesSESimpleContentItem();
		fillMapOfPropertiesSESimpleContentItem();
		fillMapOfLinksSESimpleContentItem();
	}


	private static void fillMapOfSuperNamesSEMarkedContent() {
		mapOfSuperNames.put("SEMarkedContent","SEContentItem");
	}

	private static void fillMapOfPropertiesSEMarkedContent() {
		properties = new ArrayList<String>();
		properties.add("tag");
		properties.add("structureTag");
		properties.add("E");
		properties.add("Alt");
		properties.add("ActualText");
		properties.add("inheritedLang");
		mapOfProperties.put("SEMarkedContent",properties);
	}

	private static void fillMapOfLinksSEMarkedContent() {
		links = new ArrayList<String>();
		links.add("Lang");
		mapOfLinks.put("SEMarkedContent",links);
	}

	static {
		fillMapOfSuperNamesSEMarkedContent();
		fillMapOfPropertiesSEMarkedContent();
		fillMapOfLinksSEMarkedContent();
	}


	private static void fillMapOfSuperNamesSEUnmarkedContent() {
		mapOfSuperNames.put("SEUnmarkedContent","SEContentItem");
	}

	private static void fillMapOfPropertiesSEUnmarkedContent() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEUnmarkedContent",properties);
	}

	private static void fillMapOfLinksSEUnmarkedContent() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEUnmarkedContent",links);
	}

	static {
		fillMapOfSuperNamesSEUnmarkedContent();
		fillMapOfPropertiesSEUnmarkedContent();
		fillMapOfLinksSEUnmarkedContent();
	}


	private static void fillMapOfSuperNamesSETextItem() {
		mapOfSuperNames.put("SETextItem","SESimpleContentItem");
	}

	private static void fillMapOfPropertiesSETextItem() {
		properties = new ArrayList<String>();
		properties.add("fontName");
		properties.add("scaleFactor");
		properties.add("Lang");
		mapOfProperties.put("SETextItem",properties);
	}

	private static void fillMapOfLinksSETextItem() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETextItem",links);
	}

	static {
		fillMapOfSuperNamesSETextItem();
		fillMapOfPropertiesSETextItem();
		fillMapOfLinksSETextItem();
	}


	private static void fillMapOfSuperNamesSELineArtItem() {
		mapOfSuperNames.put("SELineArtItem","SESimpleContentItem");
	}

	private static void fillMapOfPropertiesSELineArtItem() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SELineArtItem",properties);
	}

	private static void fillMapOfLinksSELineArtItem() {
		links = new ArrayList<String>();
		mapOfLinks.put("SELineArtItem",links);
	}

	static {
		fillMapOfSuperNamesSELineArtItem();
		fillMapOfPropertiesSELineArtItem();
		fillMapOfLinksSELineArtItem();
	}


	private static void fillMapOfSuperNamesSEShadingItem() {
		mapOfSuperNames.put("SEShadingItem","SESimpleContentItem");
	}

	private static void fillMapOfPropertiesSEShadingItem() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEShadingItem",properties);
	}

	private static void fillMapOfLinksSEShadingItem() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEShadingItem",links);
	}

	static {
		fillMapOfSuperNamesSEShadingItem();
		fillMapOfPropertiesSEShadingItem();
		fillMapOfLinksSEShadingItem();
	}


	private static void fillMapOfSuperNamesSEImageItem() {
		mapOfSuperNames.put("SEImageItem","SESimpleContentItem");
	}

	private static void fillMapOfPropertiesSEImageItem() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEImageItem",properties);
	}

	private static void fillMapOfLinksSEImageItem() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEImageItem",links);
	}

	static {
		fillMapOfSuperNamesSEImageItem();
		fillMapOfPropertiesSEImageItem();
		fillMapOfLinksSEImageItem();
	}


	private static void fillMapOfSuperNamesSEInlineImageItem() {
		mapOfSuperNames.put("SEInlineImageItem","SEImageItem");
	}

	private static void fillMapOfPropertiesSEInlineImageItem() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEInlineImageItem",properties);
	}

	private static void fillMapOfLinksSEInlineImageItem() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEInlineImageItem",links);
	}

	static {
		fillMapOfSuperNamesSEInlineImageItem();
		fillMapOfPropertiesSEInlineImageItem();
		fillMapOfLinksSEInlineImageItem();
	}


	private static void fillMapOfSuperNamesSEImageXObjectItem() {
		mapOfSuperNames.put("SEImageXObjectItem","SEImageItem");
	}

	private static void fillMapOfPropertiesSEImageXObjectItem() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEImageXObjectItem",properties);
	}

	private static void fillMapOfLinksSEImageXObjectItem() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEImageXObjectItem",links);
	}

	static {
		fillMapOfSuperNamesSEImageXObjectItem();
		fillMapOfPropertiesSEImageXObjectItem();
		fillMapOfLinksSEImageXObjectItem();
	}


	private static void fillMapOfSuperNamesSEForm() {
		mapOfSuperNames.put("SEForm","PDStructElem");
	}

	private static void fillMapOfPropertiesSEForm() {
		properties = new ArrayList<String>();
		properties.add("roleAttribute");
		properties.add("hasOneInteractiveChild");
		mapOfProperties.put("SEForm",properties);
	}

	private static void fillMapOfLinksSEForm() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEForm",links);
	}

	static {
		fillMapOfSuperNamesSEForm();
		fillMapOfPropertiesSEForm();
		fillMapOfLinksSEForm();
	}


	private static void fillMapOfSuperNamesSETHead() {
		mapOfSuperNames.put("SETHead","PDStructElem");
	}

	private static void fillMapOfPropertiesSETHead() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SETHead",properties);
	}

	private static void fillMapOfLinksSETHead() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETHead",links);
	}

	static {
		fillMapOfSuperNamesSETHead();
		fillMapOfPropertiesSETHead();
		fillMapOfLinksSETHead();
	}


	private static void fillMapOfSuperNamesSETBody() {
		mapOfSuperNames.put("SETBody","PDStructElem");
	}

	private static void fillMapOfPropertiesSETBody() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SETBody",properties);
	}

	private static void fillMapOfLinksSETBody() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETBody",links);
	}

	static {
		fillMapOfSuperNamesSETBody();
		fillMapOfPropertiesSETBody();
		fillMapOfLinksSETBody();
	}


	private static void fillMapOfSuperNamesSETFoot() {
		mapOfSuperNames.put("SETFoot","PDStructElem");
	}

	private static void fillMapOfPropertiesSETFoot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SETFoot",properties);
	}

	private static void fillMapOfLinksSETFoot() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETFoot",links);
	}

	static {
		fillMapOfSuperNamesSETFoot();
		fillMapOfPropertiesSETFoot();
		fillMapOfLinksSETFoot();
	}


	private static void fillMapOfSuperNamesSEAnnot() {
		mapOfSuperNames.put("SEAnnot","PDStructElem");
	}

	private static void fillMapOfPropertiesSEAnnot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEAnnot",properties);
	}

	private static void fillMapOfLinksSEAnnot() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEAnnot",links);
	}

	static {
		fillMapOfSuperNamesSEAnnot();
		fillMapOfPropertiesSEAnnot();
		fillMapOfLinksSEAnnot();
	}


	private static void fillMapOfSuperNamesSERuby() {
		mapOfSuperNames.put("SERuby","PDStructElem");
	}

	private static void fillMapOfPropertiesSERuby() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SERuby",properties);
	}

	private static void fillMapOfLinksSERuby() {
		links = new ArrayList<String>();
		mapOfLinks.put("SERuby",links);
	}

	static {
		fillMapOfSuperNamesSERuby();
		fillMapOfPropertiesSERuby();
		fillMapOfLinksSERuby();
	}


	private static void fillMapOfSuperNamesSEWarichu() {
		mapOfSuperNames.put("SEWarichu","PDStructElem");
	}

	private static void fillMapOfPropertiesSEWarichu() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEWarichu",properties);
	}

	private static void fillMapOfLinksSEWarichu() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEWarichu",links);
	}

	static {
		fillMapOfSuperNamesSEWarichu();
		fillMapOfPropertiesSEWarichu();
		fillMapOfLinksSEWarichu();
	}


	private static void fillMapOfSuperNamesSERB() {
		mapOfSuperNames.put("SERB","PDStructElem");
	}

	private static void fillMapOfPropertiesSERB() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SERB",properties);
	}

	private static void fillMapOfLinksSERB() {
		links = new ArrayList<String>();
		mapOfLinks.put("SERB",links);
	}

	static {
		fillMapOfSuperNamesSERB();
		fillMapOfPropertiesSERB();
		fillMapOfLinksSERB();
	}


	private static void fillMapOfSuperNamesSERT() {
		mapOfSuperNames.put("SERT","PDStructElem");
	}

	private static void fillMapOfPropertiesSERT() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SERT",properties);
	}

	private static void fillMapOfLinksSERT() {
		links = new ArrayList<String>();
		mapOfLinks.put("SERT",links);
	}

	static {
		fillMapOfSuperNamesSERT();
		fillMapOfPropertiesSERT();
		fillMapOfLinksSERT();
	}


	private static void fillMapOfSuperNamesSERP() {
		mapOfSuperNames.put("SERP","PDStructElem");
	}

	private static void fillMapOfPropertiesSERP() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SERP",properties);
	}

	private static void fillMapOfLinksSERP() {
		links = new ArrayList<String>();
		mapOfLinks.put("SERP",links);
	}

	static {
		fillMapOfSuperNamesSERP();
		fillMapOfPropertiesSERP();
		fillMapOfLinksSERP();
	}


	private static void fillMapOfSuperNamesSEWT() {
		mapOfSuperNames.put("SEWT","PDStructElem");
	}

	private static void fillMapOfPropertiesSEWT() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEWT",properties);
	}

	private static void fillMapOfLinksSEWT() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEWT",links);
	}

	static {
		fillMapOfSuperNamesSEWT();
		fillMapOfPropertiesSEWT();
		fillMapOfLinksSEWT();
	}


	private static void fillMapOfSuperNamesSEWP() {
		mapOfSuperNames.put("SEWP","PDStructElem");
	}

	private static void fillMapOfPropertiesSEWP() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEWP",properties);
	}

	private static void fillMapOfLinksSEWP() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEWP",links);
	}

	static {
		fillMapOfSuperNamesSEWP();
		fillMapOfPropertiesSEWP();
		fillMapOfLinksSEWP();
	}


	private static void fillMapOfSuperNamesSEDocumentFragment() {
		mapOfSuperNames.put("SEDocumentFragment","PDStructElem");
	}

	private static void fillMapOfPropertiesSEDocumentFragment() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEDocumentFragment",properties);
	}

	private static void fillMapOfLinksSEDocumentFragment() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEDocumentFragment",links);
	}

	static {
		fillMapOfSuperNamesSEDocumentFragment();
		fillMapOfPropertiesSEDocumentFragment();
		fillMapOfLinksSEDocumentFragment();
	}


	private static void fillMapOfSuperNamesSEAside() {
		mapOfSuperNames.put("SEAside","PDStructElem");
	}

	private static void fillMapOfPropertiesSEAside() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEAside",properties);
	}

	private static void fillMapOfLinksSEAside() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEAside",links);
	}

	static {
		fillMapOfSuperNamesSEAside();
		fillMapOfPropertiesSEAside();
		fillMapOfLinksSEAside();
	}


	private static void fillMapOfSuperNamesSETitle() {
		mapOfSuperNames.put("SETitle","PDStructElem");
	}

	private static void fillMapOfPropertiesSETitle() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SETitle",properties);
	}

	private static void fillMapOfLinksSETitle() {
		links = new ArrayList<String>();
		mapOfLinks.put("SETitle",links);
	}

	static {
		fillMapOfSuperNamesSETitle();
		fillMapOfPropertiesSETitle();
		fillMapOfLinksSETitle();
	}


	private static void fillMapOfSuperNamesSEFENote() {
		mapOfSuperNames.put("SEFENote","PDStructElem");
	}

	private static void fillMapOfPropertiesSEFENote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEFENote",properties);
	}

	private static void fillMapOfLinksSEFENote() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEFENote",links);
	}

	static {
		fillMapOfSuperNamesSEFENote();
		fillMapOfPropertiesSEFENote();
		fillMapOfLinksSEFENote();
	}


	private static void fillMapOfSuperNamesSESub() {
		mapOfSuperNames.put("SESub","PDStructElem");
	}

	private static void fillMapOfPropertiesSESub() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SESub",properties);
	}

	private static void fillMapOfLinksSESub() {
		links = new ArrayList<String>();
		mapOfLinks.put("SESub",links);
	}

	static {
		fillMapOfSuperNamesSESub();
		fillMapOfPropertiesSESub();
		fillMapOfLinksSESub();
	}


	private static void fillMapOfSuperNamesSEEm() {
		mapOfSuperNames.put("SEEm","PDStructElem");
	}

	private static void fillMapOfPropertiesSEEm() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEEm",properties);
	}

	private static void fillMapOfLinksSEEm() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEEm",links);
	}

	static {
		fillMapOfSuperNamesSEEm();
		fillMapOfPropertiesSEEm();
		fillMapOfLinksSEEm();
	}


	private static void fillMapOfSuperNamesSEStrong() {
		mapOfSuperNames.put("SEStrong","PDStructElem");
	}

	private static void fillMapOfPropertiesSEStrong() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEStrong",properties);
	}

	private static void fillMapOfLinksSEStrong() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEStrong",links);
	}

	static {
		fillMapOfSuperNamesSEStrong();
		fillMapOfPropertiesSEStrong();
		fillMapOfLinksSEStrong();
	}


	private static void fillMapOfSuperNamesSEArtifact() {
		mapOfSuperNames.put("SEArtifact","PDStructElem");
	}

	private static void fillMapOfPropertiesSEArtifact() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SEArtifact",properties);
	}

	private static void fillMapOfLinksSEArtifact() {
		links = new ArrayList<String>();
		mapOfLinks.put("SEArtifact",links);
	}

	static {
		fillMapOfSuperNamesSEArtifact();
		fillMapOfPropertiesSEArtifact();
		fillMapOfLinksSEArtifact();
		}


	private static void fillMapOfSuperNamesCosObject() {
		mapOfSuperNames.put("CosObject","Object");
	}

	private static void fillMapOfPropertiesCosObject() {
		properties = new ArrayList<String>();
		mapOfProperties.put("CosObject",properties);
	}

	private static void fillMapOfLinksCosObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosObject",links);
	}

	static {
		fillMapOfSuperNamesCosObject();
		fillMapOfPropertiesCosObject();
		fillMapOfLinksCosObject();
	}


	private static void fillMapOfSuperNamesCosDocument() {
		mapOfSuperNames.put("CosDocument","CosObject");
	}

	private static void fillMapOfPropertiesCosDocument() {
		properties = new ArrayList<String>();
		properties.add("nrIndirects");
		properties.add("headerVersion");
		properties.add("isOptionalContentPresent");
		properties.add("postEOFDataSize");
		properties.add("doesInfoMatchXMP");
		properties.add("firstPageID");
		properties.add("lastID");
		properties.add("firstPageIDValue");
		properties.add("lastIDValue");
		properties.add("isLinearized");
		properties.add("headerOffset");
		properties.add("header");
		properties.add("headerByte1");
		properties.add("headerByte2");
		properties.add("headerByte3");
		properties.add("headerByte4");
		properties.add("Marked");
		properties.add("Requirements");
		properties.add("NeedsRendering");
		properties.add("containsEmbeddedFiles");
		properties.add("Suspects");
		properties.add("DisplayDocTitle");
		properties.add("containsInfo");
		properties.add("containsPieceInfo");
		properties.add("MarkInfo");
		properties.add("ViewerPreferences");
		mapOfProperties.put("CosDocument",properties);
	}

	private static void fillMapOfLinksCosDocument() {
		links = new ArrayList<String>();
		links.add("EmbeddedFiles");
		links.add("xref");
		links.add("trailer");
		links.add("indirectObjects");
		links.add("document");
		links.add("doc");
		mapOfLinks.put("CosDocument",links);
	}

	static {
		fillMapOfSuperNamesCosDocument();
		fillMapOfPropertiesCosDocument();
		fillMapOfLinksCosDocument();
	}


	private static void fillMapOfSuperNamesCosIndirect() {
		mapOfSuperNames.put("CosIndirect","CosObject");
	}

	private static void fillMapOfPropertiesCosIndirect() {
		properties = new ArrayList<String>();
		properties.add("spacingCompliesPDFA");
		mapOfProperties.put("CosIndirect",properties);
	}

	private static void fillMapOfLinksCosIndirect() {
		links = new ArrayList<String>();
		links.add("directObject");
		mapOfLinks.put("CosIndirect",links);
	}

	static {
		fillMapOfSuperNamesCosIndirect();
		fillMapOfPropertiesCosIndirect();
		fillMapOfLinksCosIndirect();
	}


	private static void fillMapOfSuperNamesCosNull() {
		mapOfSuperNames.put("CosNull","CosObject");
	}

	private static void fillMapOfPropertiesCosNull() {
		properties = new ArrayList<String>();
		mapOfProperties.put("CosNull",properties);
	}

	private static void fillMapOfLinksCosNull() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosNull",links);
	}

	static {
		fillMapOfSuperNamesCosNull();
		fillMapOfPropertiesCosNull();
		fillMapOfLinksCosNull();
	}


	private static void fillMapOfSuperNamesCosBool() {
		mapOfSuperNames.put("CosBool","CosObject");
	}

	private static void fillMapOfPropertiesCosBool() {
		properties = new ArrayList<String>();
		properties.add("value");
		mapOfProperties.put("CosBool",properties);
	}

	private static void fillMapOfLinksCosBool() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosBool",links);
	}

	static {
		fillMapOfSuperNamesCosBool();
		fillMapOfPropertiesCosBool();
		fillMapOfLinksCosBool();
	}


	private static void fillMapOfSuperNamesCosNumber() {
		mapOfSuperNames.put("CosNumber","CosObject");
	}

	private static void fillMapOfPropertiesCosNumber() {
		properties = new ArrayList<String>();
		properties.add("stringValue");
		properties.add("intValue");
		properties.add("realValue");
		mapOfProperties.put("CosNumber",properties);
	}

	private static void fillMapOfLinksCosNumber() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosNumber",links);
	}

	static {
		fillMapOfSuperNamesCosNumber();
		fillMapOfPropertiesCosNumber();
		fillMapOfLinksCosNumber();
	}


	private static void fillMapOfSuperNamesCosReal() {
		mapOfSuperNames.put("CosReal","CosNumber");
	}

	private static void fillMapOfPropertiesCosReal() {
		properties = new ArrayList<String>();
		mapOfProperties.put("CosReal",properties);
	}

	private static void fillMapOfLinksCosReal() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosReal",links);
	}

	static {
		fillMapOfSuperNamesCosReal();
		fillMapOfPropertiesCosReal();
		fillMapOfLinksCosReal();
	}


	private static void fillMapOfSuperNamesCosInteger() {
		mapOfSuperNames.put("CosInteger","CosNumber");
	}

	private static void fillMapOfPropertiesCosInteger() {
		properties = new ArrayList<String>();
		mapOfProperties.put("CosInteger",properties);
	}

	private static void fillMapOfLinksCosInteger() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosInteger",links);
	}

	static {
		fillMapOfSuperNamesCosInteger();
		fillMapOfPropertiesCosInteger();
		fillMapOfLinksCosInteger();
	}


	private static void fillMapOfSuperNamesCosName() {
		mapOfSuperNames.put("CosName","CosObject");
	}

	private static void fillMapOfPropertiesCosName() {
		properties = new ArrayList<String>();
		properties.add("internalRepresentation");
		mapOfProperties.put("CosName",properties);
	}

	private static void fillMapOfLinksCosName() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosName",links);
	}

	static {
		fillMapOfSuperNamesCosName();
		fillMapOfPropertiesCosName();
		fillMapOfLinksCosName();
	}


	private static void fillMapOfSuperNamesCosBM() {
		mapOfSuperNames.put("CosBM","CosName");
	}

	private static void fillMapOfPropertiesCosBM() {
		properties = new ArrayList<String>();
		mapOfProperties.put("CosBM",properties);
	}

	private static void fillMapOfLinksCosBM() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosBM",links);
	}

	static {
		fillMapOfSuperNamesCosBM();
		fillMapOfPropertiesCosBM();
		fillMapOfLinksCosBM();
	}


	private static void fillMapOfSuperNamesCosActualText() {
		mapOfSuperNames.put("CosActualText","CosString");
	}

	private static void fillMapOfPropertiesCosActualText() {
		properties = new ArrayList<String>();
		properties.add("containsPUA");
		mapOfProperties.put("CosActualText",properties);
	}

	private static void fillMapOfLinksCosActualText() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosActualText",links);
	}

	static {
		fillMapOfSuperNamesCosActualText();
		fillMapOfPropertiesCosActualText();
		fillMapOfLinksCosActualText();
	}


	private static void fillMapOfSuperNamesCosUnicodeName() {
		mapOfSuperNames.put("CosUnicodeName","CosName");
	}

	private static void fillMapOfPropertiesCosUnicodeName() {
		properties = new ArrayList<String>();
		properties.add("isValidUtf8");
		properties.add("unicodeValue");
		mapOfProperties.put("CosUnicodeName",properties);
	}

	private static void fillMapOfLinksCosUnicodeName() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosUnicodeName",links);
	}

	static {
		fillMapOfSuperNamesCosUnicodeName();
		fillMapOfPropertiesCosUnicodeName();
		fillMapOfLinksCosUnicodeName();
	}


	private static void fillMapOfSuperNamesCosRenderingIntent() {
		mapOfSuperNames.put("CosRenderingIntent","CosName");
	}

	private static void fillMapOfPropertiesCosRenderingIntent() {
		properties = new ArrayList<String>();
		mapOfProperties.put("CosRenderingIntent",properties);
	}

	private static void fillMapOfLinksCosRenderingIntent() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosRenderingIntent",links);
	}

	static {
		fillMapOfSuperNamesCosRenderingIntent();
		fillMapOfPropertiesCosRenderingIntent();
		fillMapOfLinksCosRenderingIntent();
	}


	private static void fillMapOfSuperNamesCosFilter() {
		mapOfSuperNames.put("CosFilter","CosName");
	}

	private static void fillMapOfPropertiesCosFilter() {
		properties = new ArrayList<String>();
		properties.add("decodeParms");
		mapOfProperties.put("CosFilter",properties);
	}

	private static void fillMapOfLinksCosFilter() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosFilter",links);
	}

	static {
		fillMapOfSuperNamesCosFilter();
		fillMapOfPropertiesCosFilter();
		fillMapOfLinksCosFilter();
	}


	private static void fillMapOfSuperNamesCosIIFilter() {
		mapOfSuperNames.put("CosIIFilter","CosName");
	}

	private static void fillMapOfPropertiesCosIIFilter() {
		properties = new ArrayList<String>();
		mapOfProperties.put("CosIIFilter",properties);
	}

	private static void fillMapOfLinksCosIIFilter() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosIIFilter",links);
	}

	static {
		fillMapOfSuperNamesCosIIFilter();
		fillMapOfPropertiesCosIIFilter();
		fillMapOfLinksCosIIFilter();
	}


	private static void fillMapOfSuperNamesCosString() {
		mapOfSuperNames.put("CosString","CosObject");
	}

	private static void fillMapOfPropertiesCosString() {
		properties = new ArrayList<String>();
		properties.add("value");
		properties.add("isHex");
		properties.add("containsOnlyHex");
		properties.add("hexCount");
		mapOfProperties.put("CosString",properties);
	}

	private static void fillMapOfLinksCosString() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosString",links);
	}

	static {
		fillMapOfSuperNamesCosString();
		fillMapOfPropertiesCosString();
		fillMapOfLinksCosString();
	}


	private static void fillMapOfSuperNamesCosTextString() {
		mapOfSuperNames.put("CosTextString","CosString");
	}

	private static void fillMapOfPropertiesCosTextString() {
		properties = new ArrayList<String>();
		properties.add("unicodeValue");
		mapOfProperties.put("CosTextString",properties);
	}

	private static void fillMapOfLinksCosTextString() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosTextString",links);
	}

	static {
		fillMapOfSuperNamesCosTextString();
		fillMapOfPropertiesCosTextString();
		fillMapOfLinksCosTextString();
	}


	private static void fillMapOfSuperNamesCosLang() {
		mapOfSuperNames.put("CosLang","CosTextString");
	}

	private static void fillMapOfPropertiesCosLang() {
		properties = new ArrayList<String>();
		mapOfProperties.put("CosLang",properties);
	}

	private static void fillMapOfLinksCosLang() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosLang",links);
	}

	static {
		fillMapOfSuperNamesCosLang();
		fillMapOfPropertiesCosLang();
		fillMapOfLinksCosLang();
	}


	private static void fillMapOfSuperNamesCosDict() {
		mapOfSuperNames.put("CosDict","CosObject");
	}

	private static void fillMapOfPropertiesCosDict() {
		properties = new ArrayList<String>();
		properties.add("size");
		properties.add("keysString");
		mapOfProperties.put("CosDict",properties);
	}

	private static void fillMapOfLinksCosDict() {
		links = new ArrayList<String>();
		links.add("keys");
		links.add("values");
		links.add("metadata");
		mapOfLinks.put("CosDict",links);
	}

	static {
		fillMapOfSuperNamesCosDict();
		fillMapOfPropertiesCosDict();
		fillMapOfLinksCosDict();
	}


	private static void fillMapOfSuperNamesCosStream() {
		mapOfSuperNames.put("CosStream","CosDict");
	}

	private static void fillMapOfPropertiesCosStream() {
		properties = new ArrayList<String>();
		properties.add("Length");
		properties.add("realLength");
		properties.add("F");
		properties.add("FFilter");
		properties.add("FDecodeParms");
		properties.add("streamKeywordCRLFCompliant");
		properties.add("endstreamKeywordEOLCompliant");
		mapOfProperties.put("CosStream",properties);
	}

	private static void fillMapOfLinksCosStream() {
		links = new ArrayList<String>();
		links.add("filters");
		mapOfLinks.put("CosStream",links);
	}

	static {
		fillMapOfSuperNamesCosStream();
		fillMapOfPropertiesCosStream();
		fillMapOfLinksCosStream();
	}


	private static void fillMapOfSuperNamesCosFileSpecification() {
		mapOfSuperNames.put("CosFileSpecification","CosDict");
	}

	private static void fillMapOfPropertiesCosFileSpecification() {
		properties = new ArrayList<String>();
		properties.add("containsEF");
		properties.add("F");
		properties.add("UF");
		properties.add("AFRelationship");
		properties.add("isAssociatedFile");
		mapOfProperties.put("CosFileSpecification",properties);
	}

	private static void fillMapOfLinksCosFileSpecification() {
		links = new ArrayList<String>();
		links.add("EF");
		mapOfLinks.put("CosFileSpecification",links);
	}

	static {
		fillMapOfSuperNamesCosFileSpecification();
		fillMapOfPropertiesCosFileSpecification();
		fillMapOfLinksCosFileSpecification();
	}


	private static void fillMapOfSuperNamesCosTrailer() {
		mapOfSuperNames.put("CosTrailer","CosDict");
	}

	private static void fillMapOfPropertiesCosTrailer() {
		properties = new ArrayList<String>();
		properties.add("isEncrypted");
		mapOfProperties.put("CosTrailer",properties);
	}

	private static void fillMapOfLinksCosTrailer() {
		links = new ArrayList<String>();
		links.add("Catalog");
		links.add("Encrypt");
		links.add("Info");
		mapOfLinks.put("CosTrailer",links);
	}

	static {
		fillMapOfSuperNamesCosTrailer();
		fillMapOfPropertiesCosTrailer();
		fillMapOfLinksCosTrailer();
	}


	private static void fillMapOfSuperNamesCosInfo() {
		mapOfSuperNames.put("CosInfo","CosDict");
	}

	private static void fillMapOfPropertiesCosInfo() {
		properties = new ArrayList<String>();
		properties.add("ModDate");
		properties.add("XMPModifyDate");
		properties.add("doModDatesMatch");
		properties.add("CreationDate");
		properties.add("XMPCreateDate");
		properties.add("doCreationDatesMatch");
		properties.add("Title");
		properties.add("XMPTitle");
		properties.add("Author");
		properties.add("XMPCreator");
		properties.add("XMPCreatorSize");
		properties.add("Subject");
		properties.add("XMPDescription");
		properties.add("Producer");
		properties.add("XMPProducer");
		properties.add("Creator");
		properties.add("XMPCreatorTool");
		properties.add("Keywords");
		properties.add("XMPKeywords");
		mapOfProperties.put("CosInfo",properties);
	}

	private static void fillMapOfLinksCosInfo() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosInfo",links);
	}

	static {
		fillMapOfSuperNamesCosInfo();
		fillMapOfPropertiesCosInfo();
		fillMapOfLinksCosInfo();
	}


	private static void fillMapOfSuperNamesCosArray() {
		mapOfSuperNames.put("CosArray","CosObject");
	}

	private static void fillMapOfPropertiesCosArray() {
		properties = new ArrayList<String>();
		properties.add("size");
		mapOfProperties.put("CosArray",properties);
	}

	private static void fillMapOfLinksCosArray() {
		links = new ArrayList<String>();
		links.add("elements");
		mapOfLinks.put("CosArray",links);
	}

	static {
		fillMapOfSuperNamesCosArray();
		fillMapOfPropertiesCosArray();
		fillMapOfLinksCosArray();
	}


	private static void fillMapOfSuperNamesCosBBox() {
		mapOfSuperNames.put("CosBBox","CosArray");
	}

	private static void fillMapOfPropertiesCosBBox() {
		properties = new ArrayList<String>();
		properties.add("top");
		properties.add("bottom");
		properties.add("left");
		properties.add("right");
		mapOfProperties.put("CosBBox",properties);
	}

	private static void fillMapOfLinksCosBBox() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosBBox",links);
	}

	static {
		fillMapOfSuperNamesCosBBox();
		fillMapOfPropertiesCosBBox();
		fillMapOfLinksCosBBox();
	}


	private static void fillMapOfSuperNamesCosXRef() {
		mapOfSuperNames.put("CosXRef","CosObject");
	}

	private static void fillMapOfPropertiesCosXRef() {
		properties = new ArrayList<String>();
		properties.add("subsectionHeaderSpaceSeparated");
		properties.add("xrefEOLMarkersComplyPDFA");
		mapOfProperties.put("CosXRef",properties);
	}

	private static void fillMapOfLinksCosXRef() {
		links = new ArrayList<String>();
		mapOfLinks.put("CosXRef",links);
	}

	static {
		fillMapOfSuperNamesCosXRef();
		fillMapOfPropertiesCosXRef();
		fillMapOfLinksCosXRef();
		}


	private static void fillMapOfSuperNamesSAObject() {
		mapOfSuperNames.put("SAObject","Object");
	}

	private static void fillMapOfPropertiesSAObject() {
		properties = new ArrayList<String>();
		properties.add("structureID");
		properties.add("errorCodes");
		properties.add("errorArguments");
		mapOfProperties.put("SAObject",properties);
	}

	private static void fillMapOfLinksSAObject() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAObject",links);
	}

	static {
		fillMapOfSuperNamesSAObject();
		fillMapOfPropertiesSAObject();
		fillMapOfLinksSAObject();
	}


	private static void fillMapOfSuperNamesSAPDFDocument() {
		mapOfSuperNames.put("SAPDFDocument","Object");
	}

	private static void fillMapOfPropertiesSAPDFDocument() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAPDFDocument",properties);
	}

	private static void fillMapOfLinksSAPDFDocument() {
		links = new ArrayList<String>();
		links.add("pages");
		links.add("repeatedCharacters");
		links.add("lists");
		links.add("StructTreeRoot");
		mapOfLinks.put("SAPDFDocument",links);
	}

	static {
		fillMapOfSuperNamesSAPDFDocument();
		fillMapOfPropertiesSAPDFDocument();
		fillMapOfLinksSAPDFDocument();
	}


	private static void fillMapOfSuperNamesSAStructTreeRoot() {
		mapOfSuperNames.put("SAStructTreeRoot","Object");
	}

	private static void fillMapOfPropertiesSAStructTreeRoot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAStructTreeRoot",properties);
	}

	private static void fillMapOfLinksSAStructTreeRoot() {
		links = new ArrayList<String>();
		links.add("children");
		mapOfLinks.put("SAStructTreeRoot",links);
	}

	static {
		fillMapOfSuperNamesSAStructTreeRoot();
		fillMapOfPropertiesSAStructTreeRoot();
		fillMapOfLinksSAStructTreeRoot();
	}


	private static void fillMapOfSuperNamesSAStructElem() {
		mapOfSuperNames.put("SAStructElem","SAObject");
	}

	private static void fillMapOfPropertiesSAStructElem() {
		properties = new ArrayList<String>();
		properties.add("valueS");
		properties.add("correctSemanticScore");
		properties.add("correctType");
		properties.add("standardType");
		properties.add("isTableElem");
		properties.add("isTableChild");
		properties.add("isListElem");
		properties.add("isListChild");
		properties.add("isLeafElem");
		properties.add("parentsStandardTypes");
		properties.add("kidsStandardTypes");
		properties.add("parentStandardType");
		properties.add("hasLowestDepthError");
		properties.add("page");
		properties.add("lastPage");
		mapOfProperties.put("SAStructElem",properties);
	}

	private static void fillMapOfLinksSAStructElem() {
		links = new ArrayList<String>();
		links.add("children");
		mapOfLinks.put("SAStructElem",links);
	}

	static {
		fillMapOfSuperNamesSAStructElem();
		fillMapOfPropertiesSAStructElem();
		fillMapOfLinksSAStructElem();
	}


	private static void fillMapOfSuperNamesSAChunk() {
		mapOfSuperNames.put("SAChunk","Object");
	}

	private static void fillMapOfPropertiesSAChunk() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAChunk",properties);
	}

	private static void fillMapOfLinksSAChunk() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAChunk",links);
	}

	static {
		fillMapOfSuperNamesSAChunk();
		fillMapOfPropertiesSAChunk();
		fillMapOfLinksSAChunk();
	}


	private static void fillMapOfSuperNamesSAPage() {
		mapOfSuperNames.put("SAPage","Object");
	}

	private static void fillMapOfPropertiesSAPage() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAPage",properties);
	}

	private static void fillMapOfLinksSAPage() {
		links = new ArrayList<String>();
		links.add("annots");
		links.add("tableBorders");
		mapOfLinks.put("SAPage",links);
	}

	static {
		fillMapOfSuperNamesSAPage();
		fillMapOfPropertiesSAPage();
		fillMapOfLinksSAPage();
	}


	private static void fillMapOfSuperNamesSAAnnotation() {
		mapOfSuperNames.put("SAAnnotation","Object");
	}

	private static void fillMapOfPropertiesSAAnnotation() {
		properties = new ArrayList<String>();
		properties.add("textValue");
		properties.add("F");
		properties.add("Contents");
		properties.add("Alt");
		properties.add("isOutsideCropBox");
		mapOfProperties.put("SAAnnotation",properties);
	}

	private static void fillMapOfLinksSAAnnotation() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAAnnotation",links);
	}

	static {
		fillMapOfSuperNamesSAAnnotation();
		fillMapOfPropertiesSAAnnotation();
		fillMapOfLinksSAAnnotation();
	}


	private static void fillMapOfSuperNamesSARepeatedCharacters() {
		mapOfSuperNames.put("SARepeatedCharacters","Object");
	}

	private static void fillMapOfPropertiesSARepeatedCharacters() {
		properties = new ArrayList<String>();
		properties.add("isNonSpace");
		properties.add("numberOfRepeatedCharacters");
		mapOfProperties.put("SARepeatedCharacters",properties);
	}

	private static void fillMapOfLinksSARepeatedCharacters() {
		links = new ArrayList<String>();
		mapOfLinks.put("SARepeatedCharacters",links);
	}

	static {
		fillMapOfSuperNamesSARepeatedCharacters();
		fillMapOfPropertiesSARepeatedCharacters();
		fillMapOfLinksSARepeatedCharacters();
	}


	private static void fillMapOfSuperNamesSALinkAnnotation() {
		mapOfSuperNames.put("SALinkAnnotation","SAAnnotation");
	}

	private static void fillMapOfPropertiesSALinkAnnotation() {
		properties = new ArrayList<String>();
		properties.add("textValueIsLink");
		properties.add("contentsIsLink");
		properties.add("altIsLink");
		mapOfProperties.put("SALinkAnnotation",properties);
	}

	private static void fillMapOfLinksSALinkAnnotation() {
		links = new ArrayList<String>();
		mapOfLinks.put("SALinkAnnotation",links);
	}

	static {
		fillMapOfSuperNamesSALinkAnnotation();
		fillMapOfPropertiesSALinkAnnotation();
		fillMapOfLinksSALinkAnnotation();
	}


	private static void fillMapOfSuperNamesSATextChunk() {
		mapOfSuperNames.put("SATextChunk","SAChunk");
	}

	private static void fillMapOfPropertiesSATextChunk() {
		properties = new ArrayList<String>();
		properties.add("textSize");
		properties.add("contrastRatio");
		properties.add("textWeight");
		properties.add("hasSpecialStyle");
		properties.add("hasSpecialBackground");
		properties.add("isUnderlined");
		properties.add("parentsStandardTypes");
		properties.add("isWhiteSpaceChunk");
		mapOfProperties.put("SATextChunk",properties);
	}

	private static void fillMapOfLinksSATextChunk() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATextChunk",links);
	}

	static {
		fillMapOfSuperNamesSATextChunk();
		fillMapOfPropertiesSATextChunk();
		fillMapOfLinksSATextChunk();
	}


	private static void fillMapOfSuperNamesSAImageChunk() {
		mapOfSuperNames.put("SAImageChunk","SAChunk");
	}

	private static void fillMapOfPropertiesSAImageChunk() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAImageChunk",properties);
	}

	private static void fillMapOfLinksSAImageChunk() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAImageChunk",links);
	}

	static {
		fillMapOfSuperNamesSAImageChunk();
		fillMapOfPropertiesSAImageChunk();
		fillMapOfLinksSAImageChunk();
	}


	private static void fillMapOfSuperNamesSALineArtChunk() {
		mapOfSuperNames.put("SALineArtChunk","SAChunk");
	}

	private static void fillMapOfPropertiesSALineArtChunk() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SALineArtChunk",properties);
	}

	private static void fillMapOfLinksSALineArtChunk() {
		links = new ArrayList<String>();
		mapOfLinks.put("SALineArtChunk",links);
	}

	static {
		fillMapOfSuperNamesSALineArtChunk();
		fillMapOfPropertiesSALineArtChunk();
		fillMapOfLinksSALineArtChunk();
	}


	private static void fillMapOfSuperNamesSATableBorder() {
		mapOfSuperNames.put("SATableBorder","Object");
	}

	private static void fillMapOfPropertiesSATableBorder() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATableBorder",properties);
	}

	private static void fillMapOfLinksSATableBorder() {
		links = new ArrayList<String>();
		links.add("rows");
		mapOfLinks.put("SATableBorder",links);
	}

	static {
		fillMapOfSuperNamesSATableBorder();
		fillMapOfPropertiesSATableBorder();
		fillMapOfLinksSATableBorder();
	}


	private static void fillMapOfSuperNamesSATableBorderRow() {
		mapOfSuperNames.put("SATableBorderRow","Object");
	}

	private static void fillMapOfPropertiesSATableBorderRow() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATableBorderRow",properties);
	}

	private static void fillMapOfLinksSATableBorderRow() {
		links = new ArrayList<String>();
		links.add("cells");
		mapOfLinks.put("SATableBorderRow",links);
	}

	static {
		fillMapOfSuperNamesSATableBorderRow();
		fillMapOfPropertiesSATableBorderRow();
		fillMapOfLinksSATableBorderRow();
	}


	private static void fillMapOfSuperNamesSATableBorderCell() {
		mapOfSuperNames.put("SATableBorderCell","SAObject");
	}

	private static void fillMapOfPropertiesSATableBorderCell() {
		properties = new ArrayList<String>();
		properties.add("colSpan");
		properties.add("rowSpan");
		mapOfProperties.put("SATableBorderCell",properties);
	}

	private static void fillMapOfLinksSATableBorderCell() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATableBorderCell",links);
	}

	static {
		fillMapOfSuperNamesSATableBorderCell();
		fillMapOfPropertiesSATableBorderCell();
		fillMapOfLinksSATableBorderCell();
	}


	private static void fillMapOfSuperNamesSAList() {
		mapOfSuperNames.put("SAList","Object");
	}

	private static void fillMapOfPropertiesSAList() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAList",properties);
	}

	private static void fillMapOfLinksSAList() {
		links = new ArrayList<String>();
		links.add("items");
		mapOfLinks.put("SAList",links);
	}

	static {
		fillMapOfSuperNamesSAList();
		fillMapOfPropertiesSAList();
		fillMapOfLinksSAList();
	}


	private static void fillMapOfSuperNamesSAListItem() {
		mapOfSuperNames.put("SAListItem","SAObject");
	}

	private static void fillMapOfPropertiesSAListItem() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAListItem",properties);
	}

	private static void fillMapOfLinksSAListItem() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAListItem",links);
	}

	static {
		fillMapOfSuperNamesSAListItem();
		fillMapOfPropertiesSAListItem();
		fillMapOfLinksSAListItem();
	}


	private static void fillMapOfSuperNamesSADocument() {
		mapOfSuperNames.put("SADocument","SAStructElem");
	}

	private static void fillMapOfPropertiesSADocument() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SADocument",properties);
	}

	private static void fillMapOfLinksSADocument() {
		links = new ArrayList<String>();
		mapOfLinks.put("SADocument",links);
	}

	static {
		fillMapOfSuperNamesSADocument();
		fillMapOfPropertiesSADocument();
		fillMapOfLinksSADocument();
	}


	private static void fillMapOfSuperNamesSAPart() {
		mapOfSuperNames.put("SAPart","SAStructElem");
	}

	private static void fillMapOfPropertiesSAPart() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAPart",properties);
	}

	private static void fillMapOfLinksSAPart() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAPart",links);
	}

	static {
		fillMapOfSuperNamesSAPart();
		fillMapOfPropertiesSAPart();
		fillMapOfLinksSAPart();
	}


	private static void fillMapOfSuperNamesSADiv() {
		mapOfSuperNames.put("SADiv","SAStructElem");
	}

	private static void fillMapOfPropertiesSADiv() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SADiv",properties);
	}

	private static void fillMapOfLinksSADiv() {
		links = new ArrayList<String>();
		mapOfLinks.put("SADiv",links);
	}

	static {
		fillMapOfSuperNamesSADiv();
		fillMapOfPropertiesSADiv();
		fillMapOfLinksSADiv();
	}


	private static void fillMapOfSuperNamesSACaption() {
		mapOfSuperNames.put("SACaption","SAStructElem");
	}

	private static void fillMapOfPropertiesSACaption() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SACaption",properties);
	}

	private static void fillMapOfLinksSACaption() {
		links = new ArrayList<String>();
		mapOfLinks.put("SACaption",links);
	}

	static {
		fillMapOfSuperNamesSACaption();
		fillMapOfPropertiesSACaption();
		fillMapOfLinksSACaption();
	}


	private static void fillMapOfSuperNamesSAH() {
		mapOfSuperNames.put("SAH","SAStructElem");
	}

	private static void fillMapOfPropertiesSAH() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAH",properties);
	}

	private static void fillMapOfLinksSAH() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAH",links);
	}

	static {
		fillMapOfSuperNamesSAH();
		fillMapOfPropertiesSAH();
		fillMapOfLinksSAH();
	}


	private static void fillMapOfSuperNamesSAP() {
		mapOfSuperNames.put("SAP","SAStructElem");
	}

	private static void fillMapOfPropertiesSAP() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAP",properties);
	}

	private static void fillMapOfLinksSAP() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAP",links);
	}

	static {
		fillMapOfSuperNamesSAP();
		fillMapOfPropertiesSAP();
		fillMapOfLinksSAP();
	}


	private static void fillMapOfSuperNamesSAL() {
		mapOfSuperNames.put("SAL","SAStructElem");
	}

	private static void fillMapOfPropertiesSAL() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAL",properties);
	}

	private static void fillMapOfLinksSAL() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAL",links);
	}

	static {
		fillMapOfSuperNamesSAL();
		fillMapOfPropertiesSAL();
		fillMapOfLinksSAL();
	}


	private static void fillMapOfSuperNamesSALI() {
		mapOfSuperNames.put("SALI","SAStructElem");
	}

	private static void fillMapOfPropertiesSALI() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SALI",properties);
	}

	private static void fillMapOfLinksSALI() {
		links = new ArrayList<String>();
		mapOfLinks.put("SALI",links);
	}

	static {
		fillMapOfSuperNamesSALI();
		fillMapOfPropertiesSALI();
		fillMapOfLinksSALI();
	}


	private static void fillMapOfSuperNamesSALbl() {
		mapOfSuperNames.put("SALbl","SAStructElem");
	}

	private static void fillMapOfPropertiesSALbl() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SALbl",properties);
	}

	private static void fillMapOfLinksSALbl() {
		links = new ArrayList<String>();
		mapOfLinks.put("SALbl",links);
	}

	static {
		fillMapOfSuperNamesSALbl();
		fillMapOfPropertiesSALbl();
		fillMapOfLinksSALbl();
	}


	private static void fillMapOfSuperNamesSALBody() {
		mapOfSuperNames.put("SALBody","SAStructElem");
	}

	private static void fillMapOfPropertiesSALBody() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SALBody",properties);
	}

	private static void fillMapOfLinksSALBody() {
		links = new ArrayList<String>();
		mapOfLinks.put("SALBody",links);
	}

	static {
		fillMapOfSuperNamesSALBody();
		fillMapOfPropertiesSALBody();
		fillMapOfLinksSALBody();
	}


	private static void fillMapOfSuperNamesSATable() {
		mapOfSuperNames.put("SATable","SAStructElem");
	}

	private static void fillMapOfPropertiesSATable() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATable",properties);
	}

	private static void fillMapOfLinksSATable() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATable",links);
	}

	static {
		fillMapOfSuperNamesSATable();
		fillMapOfPropertiesSATable();
		fillMapOfLinksSATable();
	}


	private static void fillMapOfSuperNamesSATR() {
		mapOfSuperNames.put("SATR","SAStructElem");
	}

	private static void fillMapOfPropertiesSATR() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATR",properties);
	}

	private static void fillMapOfLinksSATR() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATR",links);
	}

	static {
		fillMapOfSuperNamesSATR();
		fillMapOfPropertiesSATR();
		fillMapOfLinksSATR();
	}


	private static void fillMapOfSuperNamesSATableCell() {
		mapOfSuperNames.put("SATableCell","SAStructElem");
	}

	private static void fillMapOfPropertiesSATableCell() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATableCell",properties);
	}

	private static void fillMapOfLinksSATableCell() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATableCell",links);
	}

	static {
		fillMapOfSuperNamesSATableCell();
		fillMapOfPropertiesSATableCell();
		fillMapOfLinksSATableCell();
	}


	private static void fillMapOfSuperNamesSATH() {
		mapOfSuperNames.put("SATH","SATableCell");
	}

	private static void fillMapOfPropertiesSATH() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATH",properties);
	}

	private static void fillMapOfLinksSATH() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATH",links);
	}

	static {
		fillMapOfSuperNamesSATH();
		fillMapOfPropertiesSATH();
		fillMapOfLinksSATH();
	}


	private static void fillMapOfSuperNamesSATD() {
		mapOfSuperNames.put("SATD","SATableCell");
	}

	private static void fillMapOfPropertiesSATD() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATD",properties);
	}

	private static void fillMapOfLinksSATD() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATD",links);
	}

	static {
		fillMapOfSuperNamesSATD();
		fillMapOfPropertiesSATD();
		fillMapOfLinksSATD();
	}


	private static void fillMapOfSuperNamesSASpan() {
		mapOfSuperNames.put("SASpan","SAStructElem");
	}

	private static void fillMapOfPropertiesSASpan() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SASpan",properties);
	}

	private static void fillMapOfLinksSASpan() {
		links = new ArrayList<String>();
		mapOfLinks.put("SASpan",links);
	}

	static {
		fillMapOfSuperNamesSASpan();
		fillMapOfPropertiesSASpan();
		fillMapOfLinksSASpan();
	}


	private static void fillMapOfSuperNamesSALink() {
		mapOfSuperNames.put("SALink","SAStructElem");
	}

	private static void fillMapOfPropertiesSALink() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SALink",properties);
	}

	private static void fillMapOfLinksSALink() {
		links = new ArrayList<String>();
		mapOfLinks.put("SALink",links);
	}

	static {
		fillMapOfSuperNamesSALink();
		fillMapOfPropertiesSALink();
		fillMapOfLinksSALink();
	}


	private static void fillMapOfSuperNamesSAArt() {
		mapOfSuperNames.put("SAArt","SAStructElem");
	}

	private static void fillMapOfPropertiesSAArt() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAArt",properties);
	}

	private static void fillMapOfLinksSAArt() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAArt",links);
	}

	static {
		fillMapOfSuperNamesSAArt();
		fillMapOfPropertiesSAArt();
		fillMapOfLinksSAArt();
	}


	private static void fillMapOfSuperNamesSASect() {
		mapOfSuperNames.put("SASect","SAStructElem");
	}

	private static void fillMapOfPropertiesSASect() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SASect",properties);
	}

	private static void fillMapOfLinksSASect() {
		links = new ArrayList<String>();
		mapOfLinks.put("SASect",links);
	}

	static {
		fillMapOfSuperNamesSASect();
		fillMapOfPropertiesSASect();
		fillMapOfLinksSASect();
	}


	private static void fillMapOfSuperNamesSABlockQuote() {
		mapOfSuperNames.put("SABlockQuote","SAStructElem");
	}

	private static void fillMapOfPropertiesSABlockQuote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SABlockQuote",properties);
	}

	private static void fillMapOfLinksSABlockQuote() {
		links = new ArrayList<String>();
		mapOfLinks.put("SABlockQuote",links);
	}

	static {
		fillMapOfSuperNamesSABlockQuote();
		fillMapOfPropertiesSABlockQuote();
		fillMapOfLinksSABlockQuote();
	}


	private static void fillMapOfSuperNamesSATOC() {
		mapOfSuperNames.put("SATOC","SAStructElem");
	}

	private static void fillMapOfPropertiesSATOC() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATOC",properties);
	}

	private static void fillMapOfLinksSATOC() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATOC",links);
	}

	static {
		fillMapOfSuperNamesSATOC();
		fillMapOfPropertiesSATOC();
		fillMapOfLinksSATOC();
	}


	private static void fillMapOfSuperNamesSATOCI() {
		mapOfSuperNames.put("SATOCI","SAStructElem");
	}

	private static void fillMapOfPropertiesSATOCI() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATOCI",properties);
	}

	private static void fillMapOfLinksSATOCI() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATOCI",links);
	}

	static {
		fillMapOfSuperNamesSATOCI();
		fillMapOfPropertiesSATOCI();
		fillMapOfLinksSATOCI();
	}


	private static void fillMapOfSuperNamesSAIndex() {
		mapOfSuperNames.put("SAIndex","SAStructElem");
	}

	private static void fillMapOfPropertiesSAIndex() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAIndex",properties);
	}

	private static void fillMapOfLinksSAIndex() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAIndex",links);
	}

	static {
		fillMapOfSuperNamesSAIndex();
		fillMapOfPropertiesSAIndex();
		fillMapOfLinksSAIndex();
	}


	private static void fillMapOfSuperNamesSANonStruct() {
		mapOfSuperNames.put("SANonStruct","SAStructElem");
	}

	private static void fillMapOfPropertiesSANonStruct() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SANonStruct",properties);
	}

	private static void fillMapOfLinksSANonStruct() {
		links = new ArrayList<String>();
		mapOfLinks.put("SANonStruct",links);
	}

	static {
		fillMapOfSuperNamesSANonStruct();
		fillMapOfPropertiesSANonStruct();
		fillMapOfLinksSANonStruct();
	}


	private static void fillMapOfSuperNamesSAPrivate() {
		mapOfSuperNames.put("SAPrivate","SAStructElem");
	}

	private static void fillMapOfPropertiesSAPrivate() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAPrivate",properties);
	}

	private static void fillMapOfLinksSAPrivate() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAPrivate",links);
	}

	static {
		fillMapOfSuperNamesSAPrivate();
		fillMapOfPropertiesSAPrivate();
		fillMapOfLinksSAPrivate();
	}


	private static void fillMapOfSuperNamesSAQuote() {
		mapOfSuperNames.put("SAQuote","SAStructElem");
	}

	private static void fillMapOfPropertiesSAQuote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAQuote",properties);
	}

	private static void fillMapOfLinksSAQuote() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAQuote",links);
	}

	static {
		fillMapOfSuperNamesSAQuote();
		fillMapOfPropertiesSAQuote();
		fillMapOfLinksSAQuote();
	}


	private static void fillMapOfSuperNamesSANote() {
		mapOfSuperNames.put("SANote","SAStructElem");
	}

	private static void fillMapOfPropertiesSANote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SANote",properties);
	}

	private static void fillMapOfLinksSANote() {
		links = new ArrayList<String>();
		mapOfLinks.put("SANote",links);
	}

	static {
		fillMapOfSuperNamesSANote();
		fillMapOfPropertiesSANote();
		fillMapOfLinksSANote();
	}


	private static void fillMapOfSuperNamesSAReference() {
		mapOfSuperNames.put("SAReference","SAStructElem");
	}

	private static void fillMapOfPropertiesSAReference() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAReference",properties);
	}

	private static void fillMapOfLinksSAReference() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAReference",links);
	}

	static {
		fillMapOfSuperNamesSAReference();
		fillMapOfPropertiesSAReference();
		fillMapOfLinksSAReference();
	}


	private static void fillMapOfSuperNamesSABibEntry() {
		mapOfSuperNames.put("SABibEntry","SAStructElem");
	}

	private static void fillMapOfPropertiesSABibEntry() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SABibEntry",properties);
	}

	private static void fillMapOfLinksSABibEntry() {
		links = new ArrayList<String>();
		mapOfLinks.put("SABibEntry",links);
	}

	static {
		fillMapOfSuperNamesSABibEntry();
		fillMapOfPropertiesSABibEntry();
		fillMapOfLinksSABibEntry();
	}


	private static void fillMapOfSuperNamesSACode() {
		mapOfSuperNames.put("SACode","SAStructElem");
	}

	private static void fillMapOfPropertiesSACode() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SACode",properties);
	}

	private static void fillMapOfLinksSACode() {
		links = new ArrayList<String>();
		mapOfLinks.put("SACode",links);
	}

	static {
		fillMapOfSuperNamesSACode();
		fillMapOfPropertiesSACode();
		fillMapOfLinksSACode();
	}


	private static void fillMapOfSuperNamesSAHn() {
		mapOfSuperNames.put("SAHn","SAStructElem");
	}

	private static void fillMapOfPropertiesSAHn() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAHn",properties);
	}

	private static void fillMapOfLinksSAHn() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAHn",links);
	}

	static {
		fillMapOfSuperNamesSAHn();
		fillMapOfPropertiesSAHn();
		fillMapOfLinksSAHn();
	}


	private static void fillMapOfSuperNamesSAFigure() {
		mapOfSuperNames.put("SAFigure","SAStructElem");
	}

	private static void fillMapOfPropertiesSAFigure() {
		properties = new ArrayList<String>();
		properties.add("hasBBox");
		mapOfProperties.put("SAFigure",properties);
	}

	private static void fillMapOfLinksSAFigure() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAFigure",links);
	}

	static {
		fillMapOfSuperNamesSAFigure();
		fillMapOfPropertiesSAFigure();
		fillMapOfLinksSAFigure();
	}


	private static void fillMapOfSuperNamesSAFormula() {
		mapOfSuperNames.put("SAFormula","SAStructElem");
	}

	private static void fillMapOfPropertiesSAFormula() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAFormula",properties);
	}

	private static void fillMapOfLinksSAFormula() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAFormula",links);
	}

	static {
		fillMapOfSuperNamesSAFormula();
		fillMapOfPropertiesSAFormula();
		fillMapOfLinksSAFormula();
	}


	private static void fillMapOfSuperNamesSAForm() {
		mapOfSuperNames.put("SAForm","SAStructElem");
	}

	private static void fillMapOfPropertiesSAForm() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAForm",properties);
	}

	private static void fillMapOfLinksSAForm() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAForm",links);
	}

	static {
		fillMapOfSuperNamesSAForm();
		fillMapOfPropertiesSAForm();
		fillMapOfLinksSAForm();
	}


	private static void fillMapOfSuperNamesSATHead() {
		mapOfSuperNames.put("SATHead","SAStructElem");
	}

	private static void fillMapOfPropertiesSATHead() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATHead",properties);
	}

	private static void fillMapOfLinksSATHead() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATHead",links);
	}

	static {
		fillMapOfSuperNamesSATHead();
		fillMapOfPropertiesSATHead();
		fillMapOfLinksSATHead();
	}


	private static void fillMapOfSuperNamesSATBody() {
		mapOfSuperNames.put("SATBody","SAStructElem");
	}

	private static void fillMapOfPropertiesSATBody() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATBody",properties);
	}

	private static void fillMapOfLinksSATBody() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATBody",links);
	}

	static {
		fillMapOfSuperNamesSATBody();
		fillMapOfPropertiesSATBody();
		fillMapOfLinksSATBody();
	}


	private static void fillMapOfSuperNamesSATFoot() {
		mapOfSuperNames.put("SATFoot","SAStructElem");
	}

	private static void fillMapOfPropertiesSATFoot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATFoot",properties);
	}

	private static void fillMapOfLinksSATFoot() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATFoot",links);
	}

	static {
		fillMapOfSuperNamesSATFoot();
		fillMapOfPropertiesSATFoot();
		fillMapOfLinksSATFoot();
	}


	private static void fillMapOfSuperNamesSAAnnot() {
		mapOfSuperNames.put("SAAnnot","SAStructElem");
	}

	private static void fillMapOfPropertiesSAAnnot() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAAnnot",properties);
	}

	private static void fillMapOfLinksSAAnnot() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAAnnot",links);
	}

	static {
		fillMapOfSuperNamesSAAnnot();
		fillMapOfPropertiesSAAnnot();
		fillMapOfLinksSAAnnot();
	}


	private static void fillMapOfSuperNamesSARuby() {
		mapOfSuperNames.put("SARuby","SAStructElem");
	}

	private static void fillMapOfPropertiesSARuby() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SARuby",properties);
	}

	private static void fillMapOfLinksSARuby() {
		links = new ArrayList<String>();
		mapOfLinks.put("SARuby",links);
	}

	static {
		fillMapOfSuperNamesSARuby();
		fillMapOfPropertiesSARuby();
		fillMapOfLinksSARuby();
	}


	private static void fillMapOfSuperNamesSAWarichu() {
		mapOfSuperNames.put("SAWarichu","SAStructElem");
	}

	private static void fillMapOfPropertiesSAWarichu() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAWarichu",properties);
	}

	private static void fillMapOfLinksSAWarichu() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAWarichu",links);
	}

	static {
		fillMapOfSuperNamesSAWarichu();
		fillMapOfPropertiesSAWarichu();
		fillMapOfLinksSAWarichu();
	}


	private static void fillMapOfSuperNamesSARB() {
		mapOfSuperNames.put("SARB","SAStructElem");
	}

	private static void fillMapOfPropertiesSARB() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SARB",properties);
	}

	private static void fillMapOfLinksSARB() {
		links = new ArrayList<String>();
		mapOfLinks.put("SARB",links);
	}

	static {
		fillMapOfSuperNamesSARB();
		fillMapOfPropertiesSARB();
		fillMapOfLinksSARB();
	}


	private static void fillMapOfSuperNamesSART() {
		mapOfSuperNames.put("SART","SAStructElem");
	}

	private static void fillMapOfPropertiesSART() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SART",properties);
	}

	private static void fillMapOfLinksSART() {
		links = new ArrayList<String>();
		mapOfLinks.put("SART",links);
	}

	static {
		fillMapOfSuperNamesSART();
		fillMapOfPropertiesSART();
		fillMapOfLinksSART();
	}


	private static void fillMapOfSuperNamesSARP() {
		mapOfSuperNames.put("SARP","SAStructElem");
	}

	private static void fillMapOfPropertiesSARP() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SARP",properties);
	}

	private static void fillMapOfLinksSARP() {
		links = new ArrayList<String>();
		mapOfLinks.put("SARP",links);
	}

	static {
		fillMapOfSuperNamesSARP();
		fillMapOfPropertiesSARP();
		fillMapOfLinksSARP();
	}


	private static void fillMapOfSuperNamesSAWT() {
		mapOfSuperNames.put("SAWT","SAStructElem");
	}

	private static void fillMapOfPropertiesSAWT() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAWT",properties);
	}

	private static void fillMapOfLinksSAWT() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAWT",links);
	}

	static {
		fillMapOfSuperNamesSAWT();
		fillMapOfPropertiesSAWT();
		fillMapOfLinksSAWT();
	}


	private static void fillMapOfSuperNamesSAWP() {
		mapOfSuperNames.put("SAWP","SAStructElem");
	}

	private static void fillMapOfPropertiesSAWP() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAWP",properties);
	}

	private static void fillMapOfLinksSAWP() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAWP",links);
	}

	static {
		fillMapOfSuperNamesSAWP();
		fillMapOfPropertiesSAWP();
		fillMapOfLinksSAWP();
	}


	private static void fillMapOfSuperNamesSADocumentFragment() {
		mapOfSuperNames.put("SADocumentFragment","SAStructElem");
	}

	private static void fillMapOfPropertiesSADocumentFragment() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SADocumentFragment",properties);
	}

	private static void fillMapOfLinksSADocumentFragment() {
		links = new ArrayList<String>();
		mapOfLinks.put("SADocumentFragment",links);
	}

	static {
		fillMapOfSuperNamesSADocumentFragment();
		fillMapOfPropertiesSADocumentFragment();
		fillMapOfLinksSADocumentFragment();
	}


	private static void fillMapOfSuperNamesSAAside() {
		mapOfSuperNames.put("SAAside","SAStructElem");
	}

	private static void fillMapOfPropertiesSAAside() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAAside",properties);
	}

	private static void fillMapOfLinksSAAside() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAAside",links);
	}

	static {
		fillMapOfSuperNamesSAAside();
		fillMapOfPropertiesSAAside();
		fillMapOfLinksSAAside();
	}


	private static void fillMapOfSuperNamesSATitle() {
		mapOfSuperNames.put("SATitle","SAStructElem");
	}

	private static void fillMapOfPropertiesSATitle() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SATitle",properties);
	}

	private static void fillMapOfLinksSATitle() {
		links = new ArrayList<String>();
		mapOfLinks.put("SATitle",links);
	}

	static {
		fillMapOfSuperNamesSATitle();
		fillMapOfPropertiesSATitle();
		fillMapOfLinksSATitle();
	}


	private static void fillMapOfSuperNamesSAFENote() {
		mapOfSuperNames.put("SAFENote","SAStructElem");
	}

	private static void fillMapOfPropertiesSAFENote() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAFENote",properties);
	}

	private static void fillMapOfLinksSAFENote() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAFENote",links);
	}

	static {
		fillMapOfSuperNamesSAFENote();
		fillMapOfPropertiesSAFENote();
		fillMapOfLinksSAFENote();
	}


	private static void fillMapOfSuperNamesSASub() {
		mapOfSuperNames.put("SASub","SAStructElem");
	}

	private static void fillMapOfPropertiesSASub() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SASub",properties);
	}

	private static void fillMapOfLinksSASub() {
		links = new ArrayList<String>();
		mapOfLinks.put("SASub",links);
	}

	static {
		fillMapOfSuperNamesSASub();
		fillMapOfPropertiesSASub();
		fillMapOfLinksSASub();
	}


	private static void fillMapOfSuperNamesSAEm() {
		mapOfSuperNames.put("SAEm","SAStructElem");
	}

	private static void fillMapOfPropertiesSAEm() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAEm",properties);
	}

	private static void fillMapOfLinksSAEm() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAEm",links);
	}

	static {
		fillMapOfSuperNamesSAEm();
		fillMapOfPropertiesSAEm();
		fillMapOfLinksSAEm();
	}


	private static void fillMapOfSuperNamesSAStrong() {
		mapOfSuperNames.put("SAStrong","SAStructElem");
	}

	private static void fillMapOfPropertiesSAStrong() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAStrong",properties);
	}

	private static void fillMapOfLinksSAStrong() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAStrong",links);
	}

	static {
		fillMapOfSuperNamesSAStrong();
		fillMapOfPropertiesSAStrong();
		fillMapOfLinksSAStrong();
	}


	private static void fillMapOfSuperNamesSAArtifact() {
		mapOfSuperNames.put("SAArtifact","SAStructElem");
	}

	private static void fillMapOfPropertiesSAArtifact() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SAArtifact",properties);
	}

	private static void fillMapOfLinksSAArtifact() {
		links = new ArrayList<String>();
		mapOfLinks.put("SAArtifact",links);
	}

	static {
		fillMapOfSuperNamesSAArtifact();
		fillMapOfPropertiesSAArtifact();
		fillMapOfLinksSAArtifact();
	}


	private static void fillMapOfSuperNamesSANonStandard() {
		mapOfSuperNames.put("SANonStandard","SAStructElem");
	}

	private static void fillMapOfPropertiesSANonStandard() {
		properties = new ArrayList<String>();
		mapOfProperties.put("SANonStandard",properties);
	}

	private static void fillMapOfLinksSANonStandard() {
		links = new ArrayList<String>();
		mapOfLinks.put("SANonStandard",links);
	}

	static {
		fillMapOfSuperNamesSANonStandard();
		fillMapOfPropertiesSANonStandard();
		fillMapOfLinksSANonStandard();
	}

}
