/**
 * This file is part of veraPDF Validation Model API, a module of the veraPDF project.
 * Copyright (c) 2015-2025, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF Validation Model API is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF Validation Model API as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF Validation Model API as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.model;

import org.verapdf.model.baselayer.Object;
import java.util.*;
import java.util.stream.Collectors;

public abstract class GenericModelObject implements Object {
	
	protected Boolean contextDependent = false;
	private final String objectType;
	
	protected GenericModelObject(String objectType) {
		this.objectType = objectType;
	}
	
	/**
	* @return type of the current object
	*/
	public final String getObjectType() {
		return this.objectType;
	}
	
	/**
	* @return id of the current object
	*/
	public String getID() {
		return null;
	}

	/**
	* @return extra context of the current object
	*/
	public String getExtraContext() {
		return null;
	}

	/**
	* @return context of the current object
	*/
	public String getContext() {
		return null;
	}
	
	/**
	* @param link - the name of a link
	* @return List of objects with the given link
	*/
	@Override
	public List<? extends Object> getLinkedObjects(String link) {
        throw new IllegalAccessError(this.getObjectType() + " has not access to this method or has not " + link + " link.");
    }

	/**
	* @return List of names of links for {@code this} object
	*/
    @Override
    public List<String> getLinks() {
        return ModelHelper.getListOfLinks(this.getObjectType());
    }

	/**
	* @return List of names of properties for {@code this} object
	*/
    @Override
    public List<String> getProperties() {
        return ModelHelper.getListOfProperties(this.getObjectType());
    }

	/**
	* @return null, if we have not know yet is this object context dependet of not. true, if this object is context dependent. false, if this object is not context dependent.
	*/
    @Override
    public Boolean isContextDependent() {
        return contextDependent;
    }

	/**
	* @return List of supernames for {@code this} object
	*/
    @Override
    public List<String> getSuperTypes() {
        return ModelHelper.getListOfSuperNames(this.getObjectType());
    }
}
