/**
 * This file is part of veraPDF Validation, a module of the veraPDF project.
 * Copyright (c) 2015-2024, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF Validation is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF Validation as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF Validation as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.gf.model.impl.arlington;

import org.verapdf.cos.*;
import org.verapdf.model.alayer.*;
import org.verapdf.gf.model.impl.containers.StaticContainers;
import org.verapdf.tools.StaticResources;
import java.util.*;
import org.verapdf.pd.PDNameTreeNode;
import org.verapdf.as.ASAtom;
import java.util.stream.Collectors;
import org.verapdf.pd.structure.PDNumberTreeNode;

public class GFANamespace extends GFAObject implements ANamespace {

	public GFANamespace(COSBase baseObject, COSBase parentObject, String keyName) {
		super(baseObject, parentObject, keyName, "ANamespace");
	}

	@Override
	public List<? extends org.verapdf.model.baselayer.Object> getLinkedObjects(String link) {
		switch (link) {
			case "RoleMapNS":
				return getRoleMapNS();
			case "Schema":
				return getSchema();
			default:
				return super.getLinkedObjects(link);
		}
	}

	private List<ARoleMapNS> getRoleMapNS() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON2_0:
				return getRoleMapNS2_0();
			default:
				return Collections.emptyList();
		}
	}

	private List<ARoleMapNS> getRoleMapNS2_0() {
		COSObject object = getRoleMapNSValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<ARoleMapNS> list = new ArrayList<>(1);
			list.add(new GFARoleMapNS((COSDictionary)object.getDirectBase(), this.baseObject, "RoleMapNS"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AFileSpecification> getSchema() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON2_0:
				return getSchema2_0();
			default:
				return Collections.emptyList();
		}
	}

	private List<AFileSpecification> getSchema2_0() {
		COSObject object = getSchemaValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<AFileSpecification> list = new ArrayList<>(1);
			list.add(new GFAFileSpecification((COSDictionary)object.getDirectBase(), this.baseObject, "Schema"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	@Override
	public Boolean getcontainsNS() {
		return this.baseObject.knownKey(ASAtom.getASAtom("NS"));
	}

	public COSObject getNSValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("NS"));
		return object;
	}

	@Override
	public String getNSType() {
		COSObject NS = getNSValue();
		return getObjectType(NS);
	}

	@Override
	public Boolean getNSHasTypeStringText() {
		COSObject NS = getNSValue();
		return getHasTypeStringText(NS);
	}

	@Override
	public Boolean getcontainsRoleMapNS() {
		return this.baseObject.knownKey(ASAtom.getASAtom("RoleMapNS"));
	}

	public COSObject getRoleMapNSValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("RoleMapNS"));
		return object;
	}

	@Override
	public String getRoleMapNSType() {
		COSObject RoleMapNS = getRoleMapNSValue();
		return getObjectType(RoleMapNS);
	}

	@Override
	public Boolean getRoleMapNSHasTypeDictionary() {
		COSObject RoleMapNS = getRoleMapNSValue();
		return getHasTypeDictionary(RoleMapNS);
	}

	@Override
	public Boolean getcontainsSchema() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Schema"));
	}

	public COSObject getSchemaValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Schema"));
		return object;
	}

	@Override
	public String getSchemaType() {
		COSObject Schema = getSchemaValue();
		return getObjectType(Schema);
	}

	@Override
	public Boolean getSchemaHasTypeDictionary() {
		COSObject Schema = getSchemaValue();
		return getHasTypeDictionary(Schema);
	}

	@Override
	public Boolean getSchemaHasTypeString() {
		COSObject Schema = getSchemaValue();
		return getHasTypeString(Schema);
	}

	@Override
	public Boolean getcontainsType() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Type"));
	}

	public COSObject getTypeValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Type"));
		return object;
	}

	@Override
	public String getTypeType() {
		COSObject Type = getTypeValue();
		return getObjectType(Type);
	}

	@Override
	public Boolean getTypeHasTypeName() {
		COSObject Type = getTypeValue();
		return getHasTypeName(Type);
	}

	@Override
	public String getTypeNameValue() {
		COSObject Type = getTypeValue();
		return getNameValue(Type);
	}

}
