/**
 * This file is part of veraPDF Validation for Arlington PDF Model, a module of the veraPDF project.
 * Copyright (c) 2015-2025, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF Validation for Arlington PDF Model is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF Validation for Arlington PDF Model as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF Validation for Arlington PDF Model as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.gf.model.impl.arlington;

import org.verapdf.cos.*;
import org.verapdf.model.alayer.*;
import org.verapdf.gf.model.impl.containers.StaticContainers;
import org.verapdf.tools.StaticResources;
import java.util.*;
import org.verapdf.pd.PDNameTreeNode;
import org.verapdf.as.ASAtom;
import java.util.stream.Collectors;
import org.verapdf.pd.structure.PDNumberTreeNode;

public class GFAAnnotFreeText extends GFAObject implements AAnnotFreeText {

	public GFAAnnotFreeText(COSBase baseObject, COSBase parentObject, String keyName) {
		super(baseObject, parentObject, keyName, "AAnnotFreeText");
	}

	@Override
	public List<? extends org.verapdf.model.baselayer.Object> getLinkedObjects(String link) {
		switch (link) {
			case "AF":
				return getAF();
			case "AP":
				return getAP();
			case "BE":
				return getBE();
			case "BS":
				return getBS();
			case "Border":
				return getBorder();
			case "C":
				return getC();
			case "CL":
				return getCL();
			case "ExData":
				return getExData();
			case "IRT":
				return getIRT();
			case "OC":
				return getOC();
			case "P":
				return getP();
			case "Popup":
				return getPopup();
			case "RC":
				return getRC();
			case "RD":
				return getRD();
			default:
				return super.getLinkedObjects(link);
		}
	}

	private List<AArrayOfAFFileSpecifications> getAF() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_7:
				if ((gethasExtensionISO_19005_3() == true)) {
					return getAF1_7();
				}
				return Collections.emptyList();
			case ARLINGTON2_0:
				return getAF1_7();
			default:
				return Collections.emptyList();
		}
	}

	private List<AArrayOfAFFileSpecifications> getAF1_7() {
		COSObject object = getAFValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AArrayOfAFFileSpecifications> list = new ArrayList<>(1);
			list.add(new GFAArrayOfAFFileSpecifications((COSArray)object.getDirectBase(), this.baseObject, "AF"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AAppearance> getAP() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_3:
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getAP1_3();
			default:
				return Collections.emptyList();
		}
	}

	private List<AAppearance> getAP1_3() {
		COSObject object = getAPValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<AAppearance> list = new ArrayList<>(1);
			list.add(new GFAAppearance((COSDictionary)object.getDirectBase(), this.baseObject, "AP"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<ABorderEffect> getBE() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getBE1_6();
			default:
				return Collections.emptyList();
		}
	}

	private List<ABorderEffect> getBE1_6() {
		COSObject object = getBEValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<ABorderEffect> list = new ArrayList<>(1);
			list.add(new GFABorderEffect((COSDictionary)object.getDirectBase(), this.baseObject, "BE"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<ABorderStyle> getBS() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_3:
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getBS1_3();
			default:
				return Collections.emptyList();
		}
	}

	private List<ABorderStyle> getBS1_3() {
		COSObject object = getBSValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<ABorderStyle> list = new ArrayList<>(1);
			list.add(new GFABorderStyle((COSDictionary)object.getDirectBase(), this.baseObject, "BS"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AArrayOf_4AnnotBorderCharacteristics> getBorder() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_3:
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getBorder1_3();
			default:
				return Collections.emptyList();
		}
	}

	private List<AArrayOf_4AnnotBorderCharacteristics> getBorder1_3() {
		COSObject object = getBorderValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AArrayOf_4AnnotBorderCharacteristics> list = new ArrayList<>(1);
			list.add(new GFAArrayOf_4AnnotBorderCharacteristics((COSArray)object.getDirectBase(), this.baseObject, "Border"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AArrayOf_4NumbersColorAnnotation> getC() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_3:
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getC1_3();
			default:
				return Collections.emptyList();
		}
	}

	private List<AArrayOf_4NumbersColorAnnotation> getC1_3() {
		COSObject object = getCValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AArrayOf_4NumbersColorAnnotation> list = new ArrayList<>(1);
			list.add(new GFAArrayOf_4NumbersColorAnnotation((COSArray)object.getDirectBase(), this.baseObject, "C"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AArrayOf_3or4CalloutLineNumbers> getCL() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getCL1_6();
			default:
				return Collections.emptyList();
		}
	}

	private List<AArrayOf_3or4CalloutLineNumbers> getCL1_6() {
		COSObject object = getCLValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AArrayOf_3or4CalloutLineNumbers> list = new ArrayList<>(1);
			list.add(new GFAArrayOf_3or4CalloutLineNumbers((COSArray)object.getDirectBase(), this.baseObject, "CL"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<org.verapdf.model.baselayer.Object> getExData() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_7:
				return getExData1_7();
			case ARLINGTON2_0:
				return getExData2_0();
			default:
				return Collections.emptyList();
		}
	}

	private List<org.verapdf.model.baselayer.Object> getExData1_7() {
		COSObject object = getExDataValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			org.verapdf.model.baselayer.Object result = getExDataDictionary1_7(object.getDirectBase(), "ExData");
			List<org.verapdf.model.baselayer.Object> list = new ArrayList<>(1);
			if (result != null) {
				list.add(result);
			}
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private org.verapdf.model.baselayer.Object getExDataDictionary1_7(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Subtype"));
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return null;
		}
		switch (subtypeValue) {
			case "Markup3D":
				return new GFAExData3DMarkup(base, this.baseObject, keyName);
			case "MarkupGeo":
				if (gethasExtensionADBE_Extn3() != true) {
					return null;
				}
				return new GFAExDataMarkupGeo(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private List<org.verapdf.model.baselayer.Object> getExData2_0() {
		COSObject object = getExDataValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			org.verapdf.model.baselayer.Object result = getExDataDictionary2_0(object.getDirectBase(), "ExData");
			List<org.verapdf.model.baselayer.Object> list = new ArrayList<>(1);
			if (result != null) {
				list.add(result);
			}
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private org.verapdf.model.baselayer.Object getExDataDictionary2_0(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Subtype"));
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return null;
		}
		switch (subtypeValue) {
			case "3DM":
				return new GFAExDataProjection(base, this.baseObject, keyName);
			case "Markup3D":
				return new GFAExData3DMarkup(base, this.baseObject, keyName);
			case "MarkupGeo":
				return new GFAExDataMarkupGeo(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private List<org.verapdf.model.baselayer.Object> getIRT() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
				return getIRT1_5();
			case ARLINGTON1_6:
				return getIRT1_6();
			case ARLINGTON1_7:
				return getIRT1_7();
			case ARLINGTON2_0:
				return getIRT2_0();
			default:
				return Collections.emptyList();
		}
	}

	private List<org.verapdf.model.baselayer.Object> getIRT1_5() {
		COSObject object = getIRTValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			org.verapdf.model.baselayer.Object result = getIRTDictionary1_5(object.getDirectBase(), "IRT");
			List<org.verapdf.model.baselayer.Object> list = new ArrayList<>(1);
			if (result != null) {
				list.add(result);
			}
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private org.verapdf.model.baselayer.Object getIRTDictionary1_5(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Subtype"));
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return null;
		}
		switch (subtypeValue) {
			case "Caret":
				return new GFAAnnotCaret(base, this.baseObject, keyName);
			case "Circle":
				return new GFAAnnotCircle(base, this.baseObject, keyName);
			case "FileAttachment":
				return new GFAAnnotFileAttachment(base, this.baseObject, keyName);
			case "FreeText":
				return new GFAAnnotFreeText(base, this.baseObject, keyName);
			case "Highlight":
				return new GFAAnnotHighlight(base, this.baseObject, keyName);
			case "Ink":
				return new GFAAnnotInk(base, this.baseObject, keyName);
			case "Line":
				return new GFAAnnotLine(base, this.baseObject, keyName);
			case "Link":
				return new GFAAnnotLink(base, this.baseObject, keyName);
			case "Movie":
				return new GFAAnnotMovie(base, this.baseObject, keyName);
			case "Polygon":
				return new GFAAnnotPolygon(base, this.baseObject, keyName);
			case "Popup":
				return new GFAAnnotPopup(base, this.baseObject, keyName);
			case "PrinterMark":
				return new GFAAnnotPrinterMark(base, this.baseObject, keyName);
			case "Screen":
				return new GFAAnnotScreen(base, this.baseObject, keyName);
			case "Sound":
				return new GFAAnnotSound(base, this.baseObject, keyName);
			case "Square":
				return new GFAAnnotSquare(base, this.baseObject, keyName);
			case "Squiggly":
				return new GFAAnnotSquiggly(base, this.baseObject, keyName);
			case "Stamp":
				return new GFAAnnotStamp(base, this.baseObject, keyName);
			case "StrikeOut":
				return new GFAAnnotStrikeOut(base, this.baseObject, keyName);
			case "Text":
				return new GFAAnnotText(base, this.baseObject, keyName);
			case "Underline":
				return new GFAAnnotUnderline(base, this.baseObject, keyName);
			case "Widget":
				return getIRTDictionaryWidget1_5(base, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidget1_5(COSBase base, String keyName) {
		if (base != null && base.knownKey(ASAtom.getASAtom("T"))) {
			return getIRTDictionaryWidgetT1_5(base, keyName);
		}
		return new GFAAnnotWidget(base, this.baseObject, keyName);
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetT1_5(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("FT"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("FT"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetField(base, this.baseObject, keyName);
		}
		switch (subtypeValue) {
			case "Btn":
				return getIRTDictionaryWidgetTBtn1_5(base, keyName);
			case "Ch":
				return new GFAAnnotWidgetFieldChoice(base, this.baseObject, keyName);
			case "Sig":
				return new GFAAnnotWidgetFieldSig(base, this.baseObject, keyName);
			case "Tx":
				return new GFAAnnotWidgetFieldTx(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetTBtn1_5(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Ff"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("Ff"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		Long subtypeValue = subtype.getInteger();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
		}
		switch (subtypeValue.intValue() >> 16) {
			case 0:
				return getIRTDictionaryWidgetTBtn01_5(base, keyName);
			case 1:
				return new GFAAnnotWidgetFieldBtnPush(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetTBtn01_5(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Ff"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("Ff"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		Long subtypeValue = subtype.getInteger();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
		}
		switch (subtypeValue.intValue() >> 15) {
			case 0:
				return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
			case 1:
				return new GFAAnnotWidgetFieldBtnRadio(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private List<org.verapdf.model.baselayer.Object> getIRT1_6() {
		COSObject object = getIRTValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			org.verapdf.model.baselayer.Object result = getIRTDictionary1_6(object.getDirectBase(), "IRT");
			List<org.verapdf.model.baselayer.Object> list = new ArrayList<>(1);
			if (result != null) {
				list.add(result);
			}
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private org.verapdf.model.baselayer.Object getIRTDictionary1_6(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Subtype"));
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return null;
		}
		switch (subtypeValue) {
			case "3D":
				return new GFAAnnot3D(base, this.baseObject, keyName);
			case "Caret":
				return new GFAAnnotCaret(base, this.baseObject, keyName);
			case "Circle":
				return new GFAAnnotCircle(base, this.baseObject, keyName);
			case "FileAttachment":
				return new GFAAnnotFileAttachment(base, this.baseObject, keyName);
			case "FreeText":
				return new GFAAnnotFreeText(base, this.baseObject, keyName);
			case "Highlight":
				return new GFAAnnotHighlight(base, this.baseObject, keyName);
			case "Ink":
				return new GFAAnnotInk(base, this.baseObject, keyName);
			case "Line":
				return new GFAAnnotLine(base, this.baseObject, keyName);
			case "Link":
				return new GFAAnnotLink(base, this.baseObject, keyName);
			case "Movie":
				return new GFAAnnotMovie(base, this.baseObject, keyName);
			case "Polygon":
				return new GFAAnnotPolygon(base, this.baseObject, keyName);
			case "Popup":
				return new GFAAnnotPopup(base, this.baseObject, keyName);
			case "PrinterMark":
				return new GFAAnnotPrinterMark(base, this.baseObject, keyName);
			case "Screen":
				return new GFAAnnotScreen(base, this.baseObject, keyName);
			case "Sound":
				return new GFAAnnotSound(base, this.baseObject, keyName);
			case "Square":
				return new GFAAnnotSquare(base, this.baseObject, keyName);
			case "Squiggly":
				return new GFAAnnotSquiggly(base, this.baseObject, keyName);
			case "Stamp":
				return new GFAAnnotStamp(base, this.baseObject, keyName);
			case "StrikeOut":
				return new GFAAnnotStrikeOut(base, this.baseObject, keyName);
			case "Text":
				return new GFAAnnotText(base, this.baseObject, keyName);
			case "Underline":
				return new GFAAnnotUnderline(base, this.baseObject, keyName);
			case "Watermark":
				return new GFAAnnotWatermark(base, this.baseObject, keyName);
			case "Widget":
				return getIRTDictionaryWidget1_6(base, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidget1_6(COSBase base, String keyName) {
		if (base != null && base.knownKey(ASAtom.getASAtom("T"))) {
			return getIRTDictionaryWidgetT1_6(base, keyName);
		}
		return new GFAAnnotWidget(base, this.baseObject, keyName);
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetT1_6(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("FT"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("FT"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetField(base, this.baseObject, keyName);
		}
		switch (subtypeValue) {
			case "Btn":
				return getIRTDictionaryWidgetTBtn1_6(base, keyName);
			case "Ch":
				return new GFAAnnotWidgetFieldChoice(base, this.baseObject, keyName);
			case "Sig":
				return new GFAAnnotWidgetFieldSig(base, this.baseObject, keyName);
			case "Tx":
				return new GFAAnnotWidgetFieldTx(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetTBtn1_6(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Ff"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("Ff"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		Long subtypeValue = subtype.getInteger();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
		}
		switch (subtypeValue.intValue() >> 16) {
			case 0:
				return getIRTDictionaryWidgetTBtn01_6(base, keyName);
			case 1:
				return new GFAAnnotWidgetFieldBtnPush(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetTBtn01_6(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Ff"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("Ff"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		Long subtypeValue = subtype.getInteger();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
		}
		switch (subtypeValue.intValue() >> 15) {
			case 0:
				return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
			case 1:
				return new GFAAnnotWidgetFieldBtnRadio(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private List<org.verapdf.model.baselayer.Object> getIRT1_7() {
		COSObject object = getIRTValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			org.verapdf.model.baselayer.Object result = getIRTDictionary1_7(object.getDirectBase(), "IRT");
			List<org.verapdf.model.baselayer.Object> list = new ArrayList<>(1);
			if (result != null) {
				list.add(result);
			}
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private org.verapdf.model.baselayer.Object getIRTDictionary1_7(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Subtype"));
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return null;
		}
		switch (subtypeValue) {
			case "3D":
				return new GFAAnnot3D(base, this.baseObject, keyName);
			case "Caret":
				return new GFAAnnotCaret(base, this.baseObject, keyName);
			case "Circle":
				return new GFAAnnotCircle(base, this.baseObject, keyName);
			case "FileAttachment":
				return new GFAAnnotFileAttachment(base, this.baseObject, keyName);
			case "FreeText":
				return new GFAAnnotFreeText(base, this.baseObject, keyName);
			case "Highlight":
				return new GFAAnnotHighlight(base, this.baseObject, keyName);
			case "Ink":
				return new GFAAnnotInk(base, this.baseObject, keyName);
			case "Line":
				return new GFAAnnotLine(base, this.baseObject, keyName);
			case "Link":
				return new GFAAnnotLink(base, this.baseObject, keyName);
			case "Movie":
				return new GFAAnnotMovie(base, this.baseObject, keyName);
			case "Polygon":
				return new GFAAnnotPolygon(base, this.baseObject, keyName);
			case "Popup":
				return new GFAAnnotPopup(base, this.baseObject, keyName);
			case "PrinterMark":
				return new GFAAnnotPrinterMark(base, this.baseObject, keyName);
			case "Redact":
				return new GFAAnnotRedact(base, this.baseObject, keyName);
			case "Screen":
				return new GFAAnnotScreen(base, this.baseObject, keyName);
			case "Sound":
				return new GFAAnnotSound(base, this.baseObject, keyName);
			case "Square":
				return new GFAAnnotSquare(base, this.baseObject, keyName);
			case "Squiggly":
				return new GFAAnnotSquiggly(base, this.baseObject, keyName);
			case "Stamp":
				return new GFAAnnotStamp(base, this.baseObject, keyName);
			case "StrikeOut":
				return new GFAAnnotStrikeOut(base, this.baseObject, keyName);
			case "Text":
				return new GFAAnnotText(base, this.baseObject, keyName);
			case "Underline":
				return new GFAAnnotUnderline(base, this.baseObject, keyName);
			case "Watermark":
				return new GFAAnnotWatermark(base, this.baseObject, keyName);
			case "Widget":
				return getIRTDictionaryWidget1_7(base, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidget1_7(COSBase base, String keyName) {
		if (base != null && base.knownKey(ASAtom.getASAtom("T"))) {
			return getIRTDictionaryWidgetT1_7(base, keyName);
		}
		return new GFAAnnotWidget(base, this.baseObject, keyName);
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetT1_7(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("FT"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("FT"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetField(base, this.baseObject, keyName);
		}
		switch (subtypeValue) {
			case "Btn":
				return getIRTDictionaryWidgetTBtn1_7(base, keyName);
			case "Ch":
				return new GFAAnnotWidgetFieldChoice(base, this.baseObject, keyName);
			case "Sig":
				return new GFAAnnotWidgetFieldSig(base, this.baseObject, keyName);
			case "Tx":
				return new GFAAnnotWidgetFieldTx(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetTBtn1_7(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Ff"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("Ff"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		Long subtypeValue = subtype.getInteger();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
		}
		switch (subtypeValue.intValue() >> 16) {
			case 0:
				return getIRTDictionaryWidgetTBtn01_7(base, keyName);
			case 1:
				return new GFAAnnotWidgetFieldBtnPush(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetTBtn01_7(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Ff"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("Ff"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		Long subtypeValue = subtype.getInteger();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
		}
		switch (subtypeValue.intValue() >> 15) {
			case 0:
				return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
			case 1:
				return new GFAAnnotWidgetFieldBtnRadio(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private List<org.verapdf.model.baselayer.Object> getIRT2_0() {
		COSObject object = getIRTValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			org.verapdf.model.baselayer.Object result = getIRTDictionary2_0(object.getDirectBase(), "IRT");
			List<org.verapdf.model.baselayer.Object> list = new ArrayList<>(1);
			if (result != null) {
				list.add(result);
			}
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private org.verapdf.model.baselayer.Object getIRTDictionary2_0(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Subtype"));
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return null;
		}
		switch (subtypeValue) {
			case "3D":
				return new GFAAnnot3D(base, this.baseObject, keyName);
			case "Caret":
				return new GFAAnnotCaret(base, this.baseObject, keyName);
			case "Circle":
				return new GFAAnnotCircle(base, this.baseObject, keyName);
			case "FileAttachment":
				return new GFAAnnotFileAttachment(base, this.baseObject, keyName);
			case "FreeText":
				return new GFAAnnotFreeText(base, this.baseObject, keyName);
			case "Highlight":
				return new GFAAnnotHighlight(base, this.baseObject, keyName);
			case "Ink":
				return new GFAAnnotInk(base, this.baseObject, keyName);
			case "Line":
				return new GFAAnnotLine(base, this.baseObject, keyName);
			case "Link":
				return new GFAAnnotLink(base, this.baseObject, keyName);
			case "Movie":
				return new GFAAnnotMovie(base, this.baseObject, keyName);
			case "Polygon":
				return new GFAAnnotPolygon(base, this.baseObject, keyName);
			case "Popup":
				return new GFAAnnotPopup(base, this.baseObject, keyName);
			case "PrinterMark":
				return new GFAAnnotPrinterMark(base, this.baseObject, keyName);
			case "Projection":
				return new GFAAnnotProjection(base, this.baseObject, keyName);
			case "Redact":
				return new GFAAnnotRedact(base, this.baseObject, keyName);
			case "RichMedia":
				return new GFAAnnotRichMedia(base, this.baseObject, keyName);
			case "Screen":
				return new GFAAnnotScreen(base, this.baseObject, keyName);
			case "Sound":
				return new GFAAnnotSound(base, this.baseObject, keyName);
			case "Square":
				return new GFAAnnotSquare(base, this.baseObject, keyName);
			case "Squiggly":
				return new GFAAnnotSquiggly(base, this.baseObject, keyName);
			case "Stamp":
				return new GFAAnnotStamp(base, this.baseObject, keyName);
			case "StrikeOut":
				return new GFAAnnotStrikeOut(base, this.baseObject, keyName);
			case "Text":
				return new GFAAnnotText(base, this.baseObject, keyName);
			case "Underline":
				return new GFAAnnotUnderline(base, this.baseObject, keyName);
			case "Watermark":
				return new GFAAnnotWatermark(base, this.baseObject, keyName);
			case "Widget":
				return getIRTDictionaryWidget2_0(base, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidget2_0(COSBase base, String keyName) {
		if (base != null && base.knownKey(ASAtom.getASAtom("T"))) {
			return getIRTDictionaryWidgetT2_0(base, keyName);
		}
		return new GFAAnnotWidget(base, this.baseObject, keyName);
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetT2_0(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("FT"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("FT"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetField(base, this.baseObject, keyName);
		}
		switch (subtypeValue) {
			case "Btn":
				return getIRTDictionaryWidgetTBtn2_0(base, keyName);
			case "Ch":
				return new GFAAnnotWidgetFieldChoice(base, this.baseObject, keyName);
			case "Sig":
				return new GFAAnnotWidgetFieldSig(base, this.baseObject, keyName);
			case "Tx":
				return new GFAAnnotWidgetFieldTx(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetTBtn2_0(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Ff"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("Ff"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		Long subtypeValue = subtype.getInteger();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
		}
		switch (subtypeValue.intValue() >> 16) {
			case 0:
				return getIRTDictionaryWidgetTBtn02_0(base, keyName);
			case 1:
				return new GFAAnnotWidgetFieldBtnPush(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private org.verapdf.model.baselayer.Object getIRTDictionaryWidgetTBtn02_0(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Ff"));
		COSObject parent = base.getKey(ASAtom.getASAtom("Parent"));
		Set<COSKey> visitedKeys = new HashSet<>();
		while ((subtype == null || subtype.empty()) && (parent != null && !parent.empty())) {
			subtype = parent.getKey(ASAtom.getASAtom("Ff"));
			if (parent.getKey() != null) {
				if (visitedKeys.contains(parent.getKey())) {
					break;
				}
				visitedKeys.add(parent.getKey());
			}
			parent = parent.getKey(ASAtom.getASAtom("Parent"));
		}
		if (subtype == null) {
			return null;
		}
		Long subtypeValue = subtype.getInteger();
		if (subtypeValue == null) {
			return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
		}
		switch (subtypeValue.intValue() >> 15) {
			case 0:
				return new GFAAnnotWidgetFieldBtnCheckbox(base, this.baseObject, keyName);
			case 1:
				return new GFAAnnotWidgetFieldBtnRadio(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private List<org.verapdf.model.baselayer.Object> getOC() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getOC1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<org.verapdf.model.baselayer.Object> getOC1_5() {
		COSObject object = getOCValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			org.verapdf.model.baselayer.Object result = getOCDictionary1_5(object.getDirectBase(), "OC");
			List<org.verapdf.model.baselayer.Object> list = new ArrayList<>(1);
			if (result != null) {
				list.add(result);
			}
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private org.verapdf.model.baselayer.Object getOCDictionary1_5(COSBase base, String keyName) {
		COSObject subtype = base.getKey(ASAtom.getASAtom("Type"));
		if (subtype == null) {
			return null;
		}
		String subtypeValue = subtype.getString();
		if (subtypeValue == null) {
			return null;
		}
		switch (subtypeValue) {
			case "OCG":
				return new GFAOptContentGroup(base, this.baseObject, keyName);
			case "OCMD":
				return new GFAOptContentMembership(base, this.baseObject, keyName);
			default:
				return null;
		}
	}

	private List<APageObject> getP() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_3:
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getP1_3();
			default:
				return Collections.emptyList();
		}
	}

	private List<APageObject> getP1_3() {
		COSObject object = getPValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<APageObject> list = new ArrayList<>(1);
			list.add(new GFAPageObject((COSDictionary)object.getDirectBase(), this.baseObject, "P"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AAnnotPopup> getPopup() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_3:
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getPopup1_3();
			default:
				return Collections.emptyList();
		}
	}

	private List<AAnnotPopup> getPopup1_3() {
		COSObject object = getPopupValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<AAnnotPopup> list = new ArrayList<>(1);
			list.add(new GFAAnnotPopup((COSDictionary)object.getDirectBase(), this.baseObject, "Popup"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AStream> getRC() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getRC1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<AStream> getRC1_5() {
		COSObject object = getRCValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_STREAM) {
			List<AStream> list = new ArrayList<>(1);
			list.add(new GFAStream((COSStream)object.getDirectBase(), this.baseObject, "RC"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AArrayOf_4Numbers> getRD() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getRD1_6();
			default:
				return Collections.emptyList();
		}
	}

	private List<AArrayOf_4Numbers> getRD1_6() {
		COSObject object = getRDValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AArrayOf_4Numbers> list = new ArrayList<>(1);
			list.add(new GFAArrayOf_4Numbers((COSArray)object.getDirectBase(), this.baseObject, "RD"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	@Override
	public Boolean getcontainsAF() {
		return this.baseObject.knownKey(ASAtom.getASAtom("AF"));
	}

	public COSObject getAFValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("AF"));
		return object;
	}

	@Override
	public String getAFType() {
		COSObject AF = getAFValue();
		return getObjectType(AF);
	}

	@Override
	public Boolean getAFHasTypeArray() {
		COSObject AF = getAFValue();
		return getHasTypeArray(AF);
	}

	@Override
	public Boolean getcontainsAP() {
		return this.baseObject.knownKey(ASAtom.getASAtom("AP"));
	}

	public COSObject getAPValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("AP"));
		return object;
	}

	@Override
	public String getAPType() {
		COSObject AP = getAPValue();
		return getObjectType(AP);
	}

	@Override
	public Boolean getAPHasTypeDictionary() {
		COSObject AP = getAPValue();
		return getHasTypeDictionary(AP);
	}

	@Override
	public Boolean getcontainsAS() {
		return this.baseObject.knownKey(ASAtom.getASAtom("AS"));
	}

	public COSObject getASValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("AS"));
		return object;
	}

	@Override
	public String getASType() {
		COSObject AS = getASValue();
		return getObjectType(AS);
	}

	@Override
	public Boolean getASHasTypeName() {
		COSObject AS = getASValue();
		return getHasTypeName(AS);
	}

	@Override
	public Boolean getcontainsBE() {
		return this.baseObject.knownKey(ASAtom.getASAtom("BE"));
	}

	public COSObject getBEValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("BE"));
		return object;
	}

	@Override
	public String getBEType() {
		COSObject BE = getBEValue();
		return getObjectType(BE);
	}

	@Override
	public Boolean getBEHasTypeDictionary() {
		COSObject BE = getBEValue();
		return getHasTypeDictionary(BE);
	}

	@Override
	public Boolean getcontainsBM() {
		return this.baseObject.knownKey(ASAtom.getASAtom("BM"));
	}

	public COSObject getBMDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON2_0:
				return COSName.construct("Normal");
		}
		return null;
	}

	public COSObject getBMValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("BM"));
		if (object == null || object.empty()) {
			object = getBMDefaultValue();
		}
		return object;
	}

	@Override
	public String getBMType() {
		COSObject BM = getBMValue();
		return getObjectType(BM);
	}

	@Override
	public Boolean getBMHasTypeName() {
		COSObject BM = getBMValue();
		return getHasTypeName(BM);
	}

	@Override
	public String getBMNameValue() {
		COSObject BM = getBMValue();
		return getNameValue(BM);
	}

	@Override
	public Boolean getcontainsBS() {
		return this.baseObject.knownKey(ASAtom.getASAtom("BS"));
	}

	public COSObject getBSValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("BS"));
		return object;
	}

	@Override
	public String getBSType() {
		COSObject BS = getBSValue();
		return getObjectType(BS);
	}

	@Override
	public Boolean getBSHasTypeDictionary() {
		COSObject BS = getBSValue();
		return getHasTypeDictionary(BS);
	}

	@Override
	public Boolean getcontainsBorder() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Border"));
	}

	public COSObject getBorderValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Border"));
		return object;
	}

	@Override
	public String getBorderType() {
		COSObject Border = getBorderValue();
		return getObjectType(Border);
	}

	@Override
	public Boolean getBorderHasTypeArray() {
		COSObject Border = getBorderValue();
		return getHasTypeArray(Border);
	}

	@Override
	public Boolean getcontainsC() {
		return this.baseObject.knownKey(ASAtom.getASAtom("C"));
	}

	public COSObject getCValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("C"));
		return object;
	}

	@Override
	public String getCType() {
		COSObject C = getCValue();
		return getObjectType(C);
	}

	@Override
	public Boolean getCHasTypeArray() {
		COSObject C = getCValue();
		return getHasTypeArray(C);
	}

	@Override
	public Boolean getcontainsCA() {
		return this.baseObject.knownKey(ASAtom.getASAtom("CA"));
	}

	public COSObject getCADefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return COSReal.construct(1.0D);
		}
		return null;
	}

	public COSObject getCAValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("CA"));
		if (object == null || object.empty()) {
			object = getCADefaultValue();
		}
		return object;
	}

	@Override
	public String getCAType() {
		COSObject CA = getCAValue();
		return getObjectType(CA);
	}

	@Override
	public Boolean getCAHasTypeNumber() {
		COSObject CA = getCAValue();
		return getHasTypeNumber(CA);
	}

	@Override
	public Double getCANumberValue() {
		COSObject CA = getCAValue();
		return getNumberValue(CA);
	}

	@Override
	public Boolean getcontainsCL() {
		return this.baseObject.knownKey(ASAtom.getASAtom("CL"));
	}

	public COSObject getCLValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("CL"));
		return object;
	}

	@Override
	public String getCLType() {
		COSObject CL = getCLValue();
		return getObjectType(CL);
	}

	@Override
	public Boolean getCLHasTypeArray() {
		COSObject CL = getCLValue();
		return getHasTypeArray(CL);
	}

	@Override
	public Boolean getcontainsContents() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Contents"));
	}

	public COSObject getContentsValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Contents"));
		return object;
	}

	@Override
	public String getContentsType() {
		COSObject Contents = getContentsValue();
		return getObjectType(Contents);
	}

	@Override
	public Boolean getContentsHasTypeStringText() {
		COSObject Contents = getContentsValue();
		return getHasTypeStringText(Contents);
	}

	@Override
	public Boolean getcontainsCreationDate() {
		return this.baseObject.knownKey(ASAtom.getASAtom("CreationDate"));
	}

	public COSObject getCreationDateValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("CreationDate"));
		return object;
	}

	@Override
	public String getCreationDateType() {
		COSObject CreationDate = getCreationDateValue();
		return getObjectType(CreationDate);
	}

	@Override
	public Boolean getCreationDateHasTypeDate() {
		COSObject CreationDate = getCreationDateValue();
		return getHasTypeDate(CreationDate);
	}

	@Override
	public Boolean getcontainsDA() {
		return this.baseObject.knownKey(ASAtom.getASAtom("DA"));
	}

	public COSObject getDAValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("DA"));
		return object;
	}

	@Override
	public String getDAType() {
		COSObject DA = getDAValue();
		return getObjectType(DA);
	}

	@Override
	public Boolean getDAHasTypeStringByte() {
		COSObject DA = getDAValue();
		return getHasTypeStringByte(DA);
	}

	@Override
	public Boolean getcontainsDS() {
		return this.baseObject.knownKey(ASAtom.getASAtom("DS"));
	}

	public COSObject getDSValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("DS"));
		return object;
	}

	@Override
	public String getDSType() {
		COSObject DS = getDSValue();
		return getObjectType(DS);
	}

	@Override
	public Boolean getDSHasTypeStringText() {
		COSObject DS = getDSValue();
		return getHasTypeStringText(DS);
	}

	@Override
	public Boolean getcontainsExData() {
		return this.baseObject.knownKey(ASAtom.getASAtom("ExData"));
	}

	public COSObject getExDataValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("ExData"));
		return object;
	}

	@Override
	public String getExDataType() {
		COSObject ExData = getExDataValue();
		return getObjectType(ExData);
	}

	@Override
	public Boolean getExDataHasTypeDictionary() {
		COSObject ExData = getExDataValue();
		return getHasTypeDictionary(ExData);
	}

	@Override
	public Boolean getcontainsF() {
		return this.baseObject.knownKey(ASAtom.getASAtom("F"));
	}

	public COSObject getFDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_3:
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return COSInteger.construct(0L);
		}
		return null;
	}

	public COSObject getFValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("F"));
		if (object == null || object.empty()) {
			object = getFDefaultValue();
		}
		return object;
	}

	@Override
	public String getFType() {
		COSObject F = getFValue();
		return getObjectType(F);
	}

	@Override
	public Boolean getFHasTypeBitmask() {
		COSObject F = getFValue();
		return getHasTypeBitmask(F);
	}

	@Override
	public Long getFBitmaskValue() {
		COSObject F = getFValue();
		return getBitmaskValue(F);
	}

	@Override
	public Boolean getcontainsIRT() {
		return this.baseObject.knownKey(ASAtom.getASAtom("IRT"));
	}

	public COSObject getIRTValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("IRT"));
		return object;
	}

	@Override
	public String getIRTType() {
		COSObject IRT = getIRTValue();
		return getObjectType(IRT);
	}

	@Override
	public Boolean getIRTHasTypeDictionary() {
		COSObject IRT = getIRTValue();
		return getHasTypeDictionary(IRT);
	}

	@Override
	public Boolean getcontainsIT() {
		return this.baseObject.knownKey(ASAtom.getASAtom("IT"));
	}

	public COSObject getITDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return COSName.construct("FreeText");
		}
		return null;
	}

	public COSObject getITValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("IT"));
		if (object == null || object.empty()) {
			object = getITDefaultValue();
		}
		return object;
	}

	@Override
	public String getITType() {
		COSObject IT = getITValue();
		return getObjectType(IT);
	}

	@Override
	public Boolean getITHasTypeName() {
		COSObject IT = getITValue();
		return getHasTypeName(IT);
	}

	@Override
	public String getITNameValue() {
		COSObject IT = getITValue();
		return getNameValue(IT);
	}

	@Override
	public Boolean getcontainsLE() {
		return this.baseObject.knownKey(ASAtom.getASAtom("LE"));
	}

	public COSObject getLEDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return COSName.construct("None");
		}
		return null;
	}

	public COSObject getLEValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("LE"));
		if (object == null || object.empty()) {
			object = getLEDefaultValue();
		}
		return object;
	}

	@Override
	public String getLEType() {
		COSObject LE = getLEValue();
		return getObjectType(LE);
	}

	@Override
	public Boolean getLEHasTypeName() {
		COSObject LE = getLEValue();
		return getHasTypeName(LE);
	}

	@Override
	public String getLENameValue() {
		COSObject LE = getLEValue();
		return getNameValue(LE);
	}

	@Override
	public Boolean getcontainsLang() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Lang"));
	}

	public COSObject getLangValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Lang"));
		return object;
	}

	@Override
	public String getLangType() {
		COSObject Lang = getLangValue();
		return getObjectType(Lang);
	}

	@Override
	public Boolean getLangHasTypeStringText() {
		COSObject Lang = getLangValue();
		return getHasTypeStringText(Lang);
	}

	@Override
	public Boolean getcontainsM() {
		return this.baseObject.knownKey(ASAtom.getASAtom("M"));
	}

	public COSObject getMValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("M"));
		return object;
	}

	@Override
	public String getMType() {
		COSObject M = getMValue();
		return getObjectType(M);
	}

	@Override
	public Boolean getMHasTypeDate() {
		COSObject M = getMValue();
		return getHasTypeDate(M);
	}

	@Override
	public Boolean getMHasTypeStringText() {
		COSObject M = getMValue();
		return getHasTypeStringText(M);
	}

	@Override
	public Boolean getcontainsNM() {
		return this.baseObject.knownKey(ASAtom.getASAtom("NM"));
	}

	public COSObject getNMValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("NM"));
		return object;
	}

	@Override
	public String getNMType() {
		COSObject NM = getNMValue();
		return getObjectType(NM);
	}

	@Override
	public Boolean getNMHasTypeStringText() {
		COSObject NM = getNMValue();
		return getHasTypeStringText(NM);
	}

	@Override
	public Boolean getcontainsOC() {
		return this.baseObject.knownKey(ASAtom.getASAtom("OC"));
	}

	public COSObject getOCValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("OC"));
		return object;
	}

	@Override
	public String getOCType() {
		COSObject OC = getOCValue();
		return getObjectType(OC);
	}

	@Override
	public Boolean getOCHasTypeDictionary() {
		COSObject OC = getOCValue();
		return getHasTypeDictionary(OC);
	}

	@Override
	public Boolean getcontainsP() {
		return this.baseObject.knownKey(ASAtom.getASAtom("P"));
	}

	public COSObject getPValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("P"));
		return object;
	}

	@Override
	public Boolean getisPIndirect() {
		COSObject P = getPValue();
		return getisIndirect(P);
	}

	@Override
	public String getPType() {
		COSObject P = getPValue();
		return getObjectType(P);
	}

	@Override
	public Boolean getPHasTypeDictionary() {
		COSObject P = getPValue();
		return getHasTypeDictionary(P);
	}

	@Override
	public Boolean getcontainsPopup() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Popup"));
	}

	public COSObject getPopupValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Popup"));
		return object;
	}

	@Override
	public Boolean getisPopupIndirect() {
		COSObject Popup = getPopupValue();
		return getisIndirect(Popup);
	}

	@Override
	public String getPopupType() {
		COSObject Popup = getPopupValue();
		return getObjectType(Popup);
	}

	@Override
	public Boolean getPopupHasTypeDictionary() {
		COSObject Popup = getPopupValue();
		return getHasTypeDictionary(Popup);
	}

	@Override
	public Boolean getcontainsQ() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Q"));
	}

	public COSObject getQDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_4:
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return COSInteger.construct(0L);
		}
		return null;
	}

	public COSObject getQValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Q"));
		if (object == null || object.empty()) {
			object = getQDefaultValue();
		}
		return object;
	}

	@Override
	public String getQType() {
		COSObject Q = getQValue();
		return getObjectType(Q);
	}

	@Override
	public Boolean getQHasTypeInteger() {
		COSObject Q = getQValue();
		return getHasTypeInteger(Q);
	}

	@Override
	public Long getQIntegerValue() {
		COSObject Q = getQValue();
		return getIntegerValue(Q);
	}

	@Override
	public Boolean getcontainsRC() {
		return this.baseObject.knownKey(ASAtom.getASAtom("RC"));
	}

	public COSObject getRCValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("RC"));
		return object;
	}

	@Override
	public Boolean getisRCIndirect() {
		COSObject RC = getRCValue();
		return getisIndirect(RC);
	}

	@Override
	public String getRCType() {
		COSObject RC = getRCValue();
		return getObjectType(RC);
	}

	@Override
	public Boolean getRCHasTypeStream() {
		COSObject RC = getRCValue();
		return getHasTypeStream(RC);
	}

	@Override
	public Boolean getRCHasTypeStringText() {
		COSObject RC = getRCValue();
		return getHasTypeStringText(RC);
	}

	@Override
	public Boolean getcontainsRD() {
		return this.baseObject.knownKey(ASAtom.getASAtom("RD"));
	}

	public COSObject getRDValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("RD"));
		return object;
	}

	@Override
	public String getRDType() {
		COSObject RD = getRDValue();
		return getObjectType(RD);
	}

	@Override
	public Boolean getRDHasTypeArray() {
		COSObject RD = getRDValue();
		return getHasTypeArray(RD);
	}

	@Override
	public Long getRDArraySize() {
		COSObject RD = getRDValue();
		return getArraySize(RD);
	}

	@Override
	public Boolean getcontainsRT() {
		return this.baseObject.knownKey(ASAtom.getASAtom("RT"));
	}

	public COSObject getRTDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return COSName.construct("R");
		}
		return null;
	}

	public COSObject getRTValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("RT"));
		if (object == null || object.empty()) {
			object = getRTDefaultValue();
		}
		return object;
	}

	@Override
	public String getRTType() {
		COSObject RT = getRTValue();
		return getObjectType(RT);
	}

	@Override
	public Boolean getRTHasTypeName() {
		COSObject RT = getRTValue();
		return getHasTypeName(RT);
	}

	@Override
	public String getRTNameValue() {
		COSObject RT = getRTValue();
		return getNameValue(RT);
	}

	@Override
	public Boolean getcontainsRect() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Rect"));
	}

	public COSObject getRectValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Rect"));
		return object;
	}

	@Override
	public String getRectType() {
		COSObject Rect = getRectValue();
		return getObjectType(Rect);
	}

	@Override
	public Boolean getRectHasTypeRectangle() {
		COSObject Rect = getRectValue();
		return getHasTypeRectangle(Rect);
	}

	@Override
	public Double getRectRectHeight() {
		COSObject Rect = getRectValue();
		return getRectHeight(Rect);
	}

	@Override
	public Double getRectRectWidth() {
		COSObject Rect = getRectValue();
		return getRectWidth(Rect);
	}

	@Override
	public Boolean getcontainsStructParent() {
		return this.baseObject.knownKey(ASAtom.getASAtom("StructParent"));
	}

	public COSObject getStructParentValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("StructParent"));
		return object;
	}

	@Override
	public String getStructParentType() {
		COSObject StructParent = getStructParentValue();
		return getObjectType(StructParent);
	}

	@Override
	public Boolean getStructParentHasTypeInteger() {
		COSObject StructParent = getStructParentValue();
		return getHasTypeInteger(StructParent);
	}

	@Override
	public Boolean getcontainsSubj() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Subj"));
	}

	public COSObject getSubjValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Subj"));
		return object;
	}

	@Override
	public String getSubjType() {
		COSObject Subj = getSubjValue();
		return getObjectType(Subj);
	}

	@Override
	public Boolean getSubjHasTypeStringText() {
		COSObject Subj = getSubjValue();
		return getHasTypeStringText(Subj);
	}

	@Override
	public Boolean getcontainsSubtype() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Subtype"));
	}

	public COSObject getSubtypeValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Subtype"));
		return object;
	}

	@Override
	public String getSubtypeType() {
		COSObject Subtype = getSubtypeValue();
		return getObjectType(Subtype);
	}

	@Override
	public Boolean getSubtypeHasTypeName() {
		COSObject Subtype = getSubtypeValue();
		return getHasTypeName(Subtype);
	}

	@Override
	public String getSubtypeNameValue() {
		COSObject Subtype = getSubtypeValue();
		return getNameValue(Subtype);
	}

	@Override
	public Boolean getcontainsT() {
		return this.baseObject.knownKey(ASAtom.getASAtom("T"));
	}

	public COSObject getTValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("T"));
		return object;
	}

	@Override
	public String getTType() {
		COSObject T = getTValue();
		return getObjectType(T);
	}

	@Override
	public Boolean getTHasTypeStringText() {
		COSObject T = getTValue();
		return getHasTypeStringText(T);
	}

	@Override
	public Boolean getcontainsType() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Type"));
	}

	public COSObject getTypeValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Type"));
		return object;
	}

	@Override
	public String getTypeType() {
		COSObject Type = getTypeValue();
		return getObjectType(Type);
	}

	@Override
	public Boolean getTypeHasTypeName() {
		COSObject Type = getTypeValue();
		return getHasTypeName(Type);
	}

	@Override
	public String getTypeNameValue() {
		COSObject Type = getTypeValue();
		return getNameValue(Type);
	}

	@Override
	public Boolean getcontainsca() {
		return this.baseObject.knownKey(ASAtom.getASAtom("ca"));
	}

	public COSObject getcaDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON2_0:
				return COSReal.construct(1.0D);
		}
		return null;
	}

	public COSObject getcaValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("ca"));
		if (object == null || object.empty()) {
			object = getcaDefaultValue();
		}
		return object;
	}

	@Override
	public String getcaType() {
		COSObject ca = getcaValue();
		return getObjectType(ca);
	}

	@Override
	public Boolean getcaHasTypeNumber() {
		COSObject ca = getcaValue();
		return getHasTypeNumber(ca);
	}

	@Override
	public Double getcaNumberValue() {
		COSObject ca = getcaValue();
		return getNumberValue(ca);
	}

	public COSObject getAPDValue() {
		if (this.baseObject == null || !this.baseObject.getType().isDictionaryBased()) {
			return null;
		}
		COSObject AP = this.baseObject.getKey(ASAtom.getASAtom("AP"));
		if (AP == null || !AP.getType().isDictionaryBased()) {
			return null;
		}
		COSObject D = AP.getKey(ASAtom.getASAtom("D"));
		return D;
	}

	public COSObject getAPNValue() {
		if (this.baseObject == null || !this.baseObject.getType().isDictionaryBased()) {
			return null;
		}
		COSObject AP = this.baseObject.getKey(ASAtom.getASAtom("AP"));
		if (AP == null || !AP.getType().isDictionaryBased()) {
			return null;
		}
		COSObject N = AP.getKey(ASAtom.getASAtom("N"));
		return N;
	}

	public COSObject getAPRValue() {
		if (this.baseObject == null || !this.baseObject.getType().isDictionaryBased()) {
			return null;
		}
		COSObject AP = this.baseObject.getKey(ASAtom.getASAtom("AP"));
		if (AP == null || !AP.getType().isDictionaryBased()) {
			return null;
		}
		COSObject R = AP.getKey(ASAtom.getASAtom("R"));
		return R;
	}

	@Override
	public Double getRD0NumberValue() {
		COSObject RD = getRDValue();
		if (RD == null || RD.getType() != COSObjType.COS_ARRAY || RD.size() <= 0) {
			return null;
		}
		return new GFAArrayOf_4Numbers(RD.getDirectBase(), null, null).getentry0NumberValue();
	}

	@Override
	public Double getRD1NumberValue() {
		COSObject RD = getRDValue();
		if (RD == null || RD.getType() != COSObjType.COS_ARRAY || RD.size() <= 1) {
			return null;
		}
		return new GFAArrayOf_4Numbers(RD.getDirectBase(), null, null).getentry1NumberValue();
	}

	@Override
	public Double getRD2NumberValue() {
		COSObject RD = getRDValue();
		if (RD == null || RD.getType() != COSObjType.COS_ARRAY || RD.size() <= 2) {
			return null;
		}
		return new GFAArrayOf_4Numbers(RD.getDirectBase(), null, null).getentry2NumberValue();
	}

	@Override
	public Double getRD3NumberValue() {
		COSObject RD = getRDValue();
		if (RD == null || RD.getType() != COSObjType.COS_ARRAY || RD.size() <= 3) {
			return null;
		}
		return new GFAArrayOf_4Numbers(RD.getDirectBase(), null, null).getentry3NumberValue();
	}

	@Override
	public Boolean getAPDHasTypeDictionary() {
		COSObject APD = getAPDValue();
		return getHasTypeDictionary(APD);
	}

	@Override
	public Boolean getAPNHasTypeDictionary() {
		COSObject APN = getAPNValue();
		return getHasTypeDictionary(APN);
	}

	@Override
	public Boolean getAPRHasTypeDictionary() {
		COSObject APR = getAPRValue();
		return getHasTypeDictionary(APR);
	}

	@Override
	public Boolean getRD0HasTypeNumber() {
		COSObject RD = getRDValue();
		if (RD == null || RD.getType() != COSObjType.COS_ARRAY || RD.size() <= 0) {
			return null;
		}
		return new GFAArrayOf_4Numbers(RD.getDirectBase(), null, null).getentry0HasTypeNumber();
	}

	@Override
	public Boolean getRD1HasTypeNumber() {
		COSObject RD = getRDValue();
		if (RD == null || RD.getType() != COSObjType.COS_ARRAY || RD.size() <= 1) {
			return null;
		}
		return new GFAArrayOf_4Numbers(RD.getDirectBase(), null, null).getentry1HasTypeNumber();
	}

	@Override
	public Boolean getRD2HasTypeNumber() {
		COSObject RD = getRDValue();
		if (RD == null || RD.getType() != COSObjType.COS_ARRAY || RD.size() <= 2) {
			return null;
		}
		return new GFAArrayOf_4Numbers(RD.getDirectBase(), null, null).getentry2HasTypeNumber();
	}

	@Override
	public Boolean getRD3HasTypeNumber() {
		COSObject RD = getRDValue();
		if (RD == null || RD.getType() != COSObjType.COS_ARRAY || RD.size() <= 3) {
			return null;
		}
		return new GFAArrayOf_4Numbers(RD.getDirectBase(), null, null).getentry3HasTypeNumber();
	}

}
