/**
 * This file is part of veraPDF wcag algorithms, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF wcag algorithms is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF wcag algorithms as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF wcag algorithms as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.wcag.algorithms.entities.lists;

import org.verapdf.wcag.algorithms.entities.content.InfoChunk;
import org.verapdf.wcag.algorithms.entities.tables.Table;
import org.verapdf.wcag.algorithms.entities.tables.TableCell;
import org.verapdf.wcag.algorithms.entities.tables.TableRow;
import org.verapdf.wcag.algorithms.entities.tables.TableToken;
import org.verapdf.wcag.algorithms.semanticalgorithms.utils.TableUtils;

import java.util.ArrayList;
import java.util.List;

public class PDFList extends InfoChunk {

    private final Long id;
    private final List<ListItem> listItems;

    public PDFList(Table table) {
        super(table.getBoundingBox());
        this.id = table.getId();
        listItems = new ArrayList<>();
        createListItemsFromTableRows(table.getRows());
    }

    public int getNumberOfListItems() {
        return listItems.size();
    }

    public List<ListItem> getListItems() {
        return listItems;
    }

    public Long getId() {
        return id;
    }

    public void add(ListItem listItem) {
        listItems.add(listItem);
        getBoundingBox().union(listItem.getBoundingBox());
    }

    private void createListItemsFromTableRows(List<TableRow> tableRows) {
        ListItem lastListItem = null;
        Double maxGap = null;
        for (TableRow row : tableRows) {
            TableCell cell = row.getCells().get(0);
            if (cell.getString().trim().isEmpty() && (cell.getContent().isEmpty() ||
                    cell.getContent().get(0).getType() == TableToken.TableTokenType.TEXT) && lastListItem != null) {
                double gap = TableUtils.getRowGapFactor(lastListItem, row);
                if (maxGap == null) {
                    maxGap = TableUtils.NEXT_LINE_MAX_TOLERANCE_FACTOR * gap;
                } else if (gap < maxGap) {
                    maxGap = Math.max(TableUtils.NEXT_LINE_MAX_TOLERANCE_FACTOR * gap, maxGap);
                } else {
                    break;
                }
                lastListItem.add(row);
            } else {
                lastListItem = new ListItem(row);
                add(lastListItem);
            }
        }
    }

}
