/**
 * This file is part of veraPDF wcag algorithms, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF wcag algorithms is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF wcag algorithms as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF wcag algorithms as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.wcag.algorithms.semanticalgorithms;

import org.verapdf.wcag.algorithms.entities.IDocument;
import org.verapdf.wcag.algorithms.entities.ITree;
import org.verapdf.wcag.algorithms.entities.tables.TableBordersCollection;
import org.verapdf.wcag.algorithms.semanticalgorithms.consumers.*;
import org.verapdf.wcag.algorithms.semanticalgorithms.containers.StaticContainers;

import java.util.function.Consumer;

public class AccumulatedNodeSemanticChecker implements ISemanticsChecker {

	@Override
	public void checkSemanticDocument(IDocument document, String fileName) {
		StaticContainers.updateContainers(document);

		ITree tree = document.getTree();
		if (tree == null) {
			return;
		}

		LinesPreprocessingConsumer linesPreprocessingConsumer = new LinesPreprocessingConsumer();
		if (!startNextStep(linesPreprocessingConsumer)) {
			return;
		}
		linesPreprocessingConsumer.findTableBorders();
		StaticContainers.setTableBordersCollection(new TableBordersCollection(linesPreprocessingConsumer.getTableBorders()));

		SemanticDocumentPreprocessingConsumer semanticDocumentValidator = new SemanticDocumentPreprocessingConsumer();
		if (!startNextStep(semanticDocumentValidator)) {
			return;
		}
		tree.forEach(semanticDocumentValidator);
		StaticContainers.setStructElementsNumber(semanticDocumentValidator.getStructElementsNumber());
		StaticContainers.setTextChunksNumber(semanticDocumentValidator.getTextChunksNumber());

		if (fileName != null) {
			ContrastRatioConsumer contrastRatioConsumer = new ContrastRatioConsumer(fileName);
			if (!startNextStep(contrastRatioConsumer)) {
				return;
			}
			contrastRatioConsumer.calculateContrast(document.getTree());
		}

		AccumulatedNodeConsumer semanticDetectionValidator = new AccumulatedNodeConsumer();
		if (!startNextStep(semanticDetectionValidator)) {
			return;
		}
		tree.forEach(semanticDetectionValidator);

		HeadingCaptionConsumer headingCaptionConsumer = new HeadingCaptionConsumer();
		if (!startNextStep(headingCaptionConsumer)) {
			return;
		}
		tree.forEach(headingCaptionConsumer);

		TOCDetectionConsumer tocDetectionConsumer = new TOCDetectionConsumer();
		if (!startNextStep(tocDetectionConsumer)) {
			return;
		}
		tree.forEach(tocDetectionConsumer);

		ListDetectionConsumer listDetectionConsumer = new ListDetectionConsumer();
		if (!startNextStep(listDetectionConsumer)) {
			return;
		}
		tree.forEach(listDetectionConsumer);

		TableBorderConsumer tableBorderConsumer = new TableBorderConsumer();
		if (!startNextStep(tableBorderConsumer)) {
			return;
		}
		tableBorderConsumer.recognizeTables(tree);

		TableChecker tableChecker = new TableChecker();
		if (!startNextStep(tableChecker)) {
			return;
		}
		tree.forEach(tableChecker);

		ClusterTableConsumer tableFinder = new ClusterTableConsumer();
		if (!startNextStep(tableFinder)) {
			return;
		}
		tableFinder.findTables(tree.getRoot());

		SemanticDocumentPostprocessingConsumer documentPostprocessingConsumer = new SemanticDocumentPostprocessingConsumer();
		if (!startNextStep(documentPostprocessingConsumer)) {
			return;
		}
		documentPostprocessingConsumer.runPostprocessingChecks(tree);

		StaticContainers.getWCAGValidationInfo().setCurrentConsumer(null);
	}

	public static boolean startNextStep(WCAGConsumer consumer) {
		if (StaticContainers.getWCAGValidationInfo().getAbortProcessing()) {
			StaticContainers.getWCAGValidationInfo().setAbortProcessing(false);
			StaticContainers.getWCAGValidationInfo().setCurrentConsumer(null);
			return false;
		}
		StaticContainers.getWCAGValidationInfo().setCurrentConsumer(consumer);
		return true;
	}
}
