/**
 * This file is part of veraPDF Validation for Arlington PDF Model, a module of the veraPDF project.
 * Copyright (c) 2015-2025, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF Validation for Arlington PDF Model is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF Validation for Arlington PDF Model as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF Validation for Arlington PDF Model as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.gf.model.impl.arlington;

import org.verapdf.cos.*;
import org.verapdf.model.alayer.*;
import org.verapdf.gf.model.impl.containers.StaticContainers;
import org.verapdf.tools.StaticResources;
import java.util.*;
import org.verapdf.pd.PDNameTreeNode;
import org.verapdf.as.ASAtom;
import java.util.stream.Collectors;
import org.verapdf.pd.structure.PDNumberTreeNode;

public class GFAURTransformParameters extends GFAObject implements AURTransformParameters {

	public GFAURTransformParameters(COSBase baseObject, COSBase parentObject, String keyName) {
		super(baseObject, parentObject, keyName, "AURTransformParameters");
	}

	@Override
	public List<? extends org.verapdf.model.baselayer.Object> getLinkedObjects(String link) {
		switch (link) {
			case "Annots":
				return getAnnots();
			case "Document":
				return getDocument();
			case "EF":
				return getEF();
			case "Form":
				return getForm();
			case "Signature":
				return getSignature();
			default:
				return super.getLinkedObjects(link);
		}
	}

	private List<AURTransformParamAnnotsArray> getAnnots() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getAnnots1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<AURTransformParamAnnotsArray> getAnnots1_5() {
		COSObject object = getAnnotsValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AURTransformParamAnnotsArray> list = new ArrayList<>(1);
			list.add(new GFAURTransformParamAnnotsArray((COSArray)object.getDirectBase(), this.baseObject, "Annots"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AURTransformParamDocumentArray> getDocument() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getDocument1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<AURTransformParamDocumentArray> getDocument1_5() {
		COSObject object = getDocumentValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AURTransformParamDocumentArray> list = new ArrayList<>(1);
			list.add(new GFAURTransformParamDocumentArray((COSArray)object.getDirectBase(), this.baseObject, "Document"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AURTransformParamEFArray> getEF() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getEF1_6();
			default:
				return Collections.emptyList();
		}
	}

	private List<AURTransformParamEFArray> getEF1_6() {
		COSObject object = getEFValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AURTransformParamEFArray> list = new ArrayList<>(1);
			list.add(new GFAURTransformParamEFArray((COSArray)object.getDirectBase(), this.baseObject, "EF"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AURTransformParamFormArray> getForm() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getForm1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<AURTransformParamFormArray> getForm1_5() {
		COSObject object = getFormValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AURTransformParamFormArray> list = new ArrayList<>(1);
			list.add(new GFAURTransformParamFormArray((COSArray)object.getDirectBase(), this.baseObject, "Form"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<AURTransformParamSignatureArray> getSignature() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getSignature1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<AURTransformParamSignatureArray> getSignature1_5() {
		COSObject object = getSignatureValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_ARRAY) {
			List<AURTransformParamSignatureArray> list = new ArrayList<>(1);
			list.add(new GFAURTransformParamSignatureArray((COSArray)object.getDirectBase(), this.baseObject, "Signature"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	@Override
	public Boolean getcontainsAnnots() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Annots"));
	}

	public COSObject getAnnotsValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Annots"));
		return object;
	}

	@Override
	public String getAnnotsType() {
		COSObject Annots = getAnnotsValue();
		return getObjectType(Annots);
	}

	@Override
	public Boolean getAnnotsHasTypeArray() {
		COSObject Annots = getAnnotsValue();
		return getHasTypeArray(Annots);
	}

	@Override
	public Boolean getcontainsDocument() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Document"));
	}

	public COSObject getDocumentValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Document"));
		return object;
	}

	@Override
	public String getDocumentType() {
		COSObject Document = getDocumentValue();
		return getObjectType(Document);
	}

	@Override
	public Boolean getDocumentHasTypeArray() {
		COSObject Document = getDocumentValue();
		return getHasTypeArray(Document);
	}

	@Override
	public Boolean getcontainsEF() {
		return this.baseObject.knownKey(ASAtom.getASAtom("EF"));
	}

	public COSObject getEFValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("EF"));
		return object;
	}

	@Override
	public String getEFType() {
		COSObject EF = getEFValue();
		return getObjectType(EF);
	}

	@Override
	public Boolean getEFHasTypeArray() {
		COSObject EF = getEFValue();
		return getHasTypeArray(EF);
	}

	@Override
	public Boolean getcontainsForm() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Form"));
	}

	public COSObject getFormValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Form"));
		return object;
	}

	@Override
	public String getFormType() {
		COSObject Form = getFormValue();
		return getObjectType(Form);
	}

	@Override
	public Boolean getFormHasTypeArray() {
		COSObject Form = getFormValue();
		return getHasTypeArray(Form);
	}

	@Override
	public Boolean getcontainsMsg() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Msg"));
	}

	public COSObject getMsgValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Msg"));
		return object;
	}

	@Override
	public String getMsgType() {
		COSObject Msg = getMsgValue();
		return getObjectType(Msg);
	}

	@Override
	public Boolean getMsgHasTypeStringText() {
		COSObject Msg = getMsgValue();
		return getHasTypeStringText(Msg);
	}

	@Override
	public Boolean getcontainsP() {
		return this.baseObject.knownKey(ASAtom.getASAtom("P"));
	}

	public COSObject getPDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return COSBoolean.construct(false);
		}
		return null;
	}

	public COSObject getPValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("P"));
		if (object == null || object.empty()) {
			object = getPDefaultValue();
		}
		return object;
	}

	@Override
	public String getPType() {
		COSObject P = getPValue();
		return getObjectType(P);
	}

	@Override
	public Boolean getPHasTypeBoolean() {
		COSObject P = getPValue();
		return getHasTypeBoolean(P);
	}

	@Override
	public Boolean getcontainsSignature() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Signature"));
	}

	public COSObject getSignatureValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Signature"));
		return object;
	}

	@Override
	public String getSignatureType() {
		COSObject Signature = getSignatureValue();
		return getObjectType(Signature);
	}

	@Override
	public Boolean getSignatureHasTypeArray() {
		COSObject Signature = getSignatureValue();
		return getHasTypeArray(Signature);
	}

	@Override
	public Boolean getcontainsType() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Type"));
	}

	public COSObject getTypeValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Type"));
		return object;
	}

	@Override
	public String getTypeType() {
		COSObject Type = getTypeValue();
		return getObjectType(Type);
	}

	@Override
	public Boolean getTypeHasTypeName() {
		COSObject Type = getTypeValue();
		return getHasTypeName(Type);
	}

	@Override
	public String getTypeNameValue() {
		COSObject Type = getTypeValue();
		return getNameValue(Type);
	}

	@Override
	public Boolean getcontainsV() {
		return this.baseObject.knownKey(ASAtom.getASAtom("V"));
	}

	public COSObject getVDefaultValue() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return COSName.construct("2.2");
		}
		return null;
	}

	public COSObject getVValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("V"));
		if (object == null || object.empty()) {
			object = getVDefaultValue();
		}
		return object;
	}

	@Override
	public String getVType() {
		COSObject V = getVValue();
		return getObjectType(V);
	}

	@Override
	public Boolean getVHasTypeName() {
		COSObject V = getVValue();
		return getHasTypeName(V);
	}

}
