/**
 * This file is part of veraPDF PDF/A XMP Library Core, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF PDF/A XMP Library Core is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF PDF/A XMP Library Core as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF PDF/A XMP Library Core as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package com.adobe.xmp.impl;

import com.adobe.xmp.XMPMetaFactory;
import com.adobe.xmp.options.PropertyOptions;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Maksim Bezrukov
 */
public class VeraPDFXMPNode {

    /**
     * original <code>XMPNode</code> object from metadata
     */
    private XMPNode originalNode;
    /**
     * namespace URI of the node
     */
    private String namespaceURI;
    /**
     * prefix of the node
     */
    private String prefix;
    /**
     * name of the node, contains different information depending of the node kind
     */
    private String name;
    /**
     * value of the node, contains different information depending of the node kind
     */
    private String value;
    /**
     * list of child nodes, lazy initialized
     */
    private List<VeraPDFXMPNode> children;
    /**
     * options describing the kind of the node
     */
    private PropertyOptions options;

    /**
     * Creates an <code>VeraPDFXMPNode</code> with initial values.
     *
     * @param prefix  the prefix of the node
     * @param name    the name of the node
     * @param value   the value of the node
     * @param children list of children for the node
     * @param options the options of the node
     */
    private VeraPDFXMPNode(String prefix, String namespaceURI, String name, String value, List<VeraPDFXMPNode> children, PropertyOptions options, XMPNode originalNode) {
        this.prefix = prefix;
        this.namespaceURI = namespaceURI;
        this.name = name;
        this.value = value;
        this.children = children;
        this.options = options;
        this.originalNode = originalNode;
    }

    static VeraPDFXMPNode fromXMPNode(XMPNode original) {
        if (original == null) {
            return null;
        }
        String prefix = original.getOriginalPrefix();
        String originalName = original.getName();
        int prefixEndIndex = originalName.indexOf(":");
        String name = originalName.substring(prefixEndIndex + 1, originalName.length());
        String value = original.getValue();
        PropertyOptions options = original.getOptions();
        List originalChildren = original.getUnmodifiableChildren();
        List<VeraPDFXMPNode> children = new ArrayList<>(originalChildren.size());
        for (Object child : originalChildren) {
            children.add(fromXMPNode((XMPNode) child));
        }
        String namespaceURI = XMPMetaFactory.getSchemaRegistry().getNamespaceURI(originalName.substring(0, Math.max(prefixEndIndex, 0)));
        return new VeraPDFXMPNode(prefix, namespaceURI, name, value, children, options, original);
    }

    public String getPrefix() {
        return prefix;
    }

    public String getNamespaceURI() {
        return namespaceURI;
    }

    public String getName() {
        return name;
    }

    public String getValue() {
        return value;
    }

    public boolean isLanguageAlternative() {
        if (children.isEmpty()) {
            return false;
        }
        for(VeraPDFXMPNode node : children) {
            List elems = node.getQualifier();
            boolean hasLanguageNode = false;
            for(Object elem : elems) {
                if (((XMPNode)elem).isLanguageNode()) {
                    hasLanguageNode = true;
                }
            }
            if (!hasLanguageNode) {
                return false;
            }
        }
        return true;
    }

    public List getQualifier(){
        return originalNode.getQualifier();
    }

    public List<VeraPDFXMPNode> getChildren() {
        return children;
    }

    public PropertyOptions getOptions() {
        return options;
    }
}
