<?xml version="1.0"?>
<!--
    Transform an XML Validation Report into presentable HTML.
    Author: Maksim Bezrukov
    Version: 1.0
-->

<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

    <xsl:output indent="yes" method="html"/>

    <!-- Parameter to select a full HTML document (true) or a fragment within div (false) -->
    <xsl:param name="isFullHTML" select="'true'"/>
    <!-- Parameter to show parser type -->
    <xsl:param name="parserType"/>
    <!-- Parameter to show release details -->
    <xsl:param name="appName"/>
    <!-- Prameter to set the base path to the Wiki instance -->
    <xsl:param name="wikiPath"
               select="'https://github.com/veraPDF/veraPDF-validation-profiles/wiki/'"/>
    <xsl:strip-space elements="*"/>

    <!-- HTML header and body wrapper -->
    <xsl:template match="/">
        <xsl:if test="$isFullHTML='true'">
            <html>
                <head>
                    <title>Arlington PDF Model Report</title>
                </head>
                <body>
                    <!-- Header image and overall title -->
                    <p>
                        <img alt="arlington Logo" height="180"
                             src="data:image/png;base64,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"/>
                    </p>
                    <style>
                        <xsl:text>body{font-family: sans-serif;}</xsl:text>
                    </style>
                    <h1 align="left">
                        <strong>
                            <b>Arlington PDF Model Report</b>
                        </strong>
                    </h1>
                    <xsl:apply-templates/>
                </body>
            </html>
        </xsl:if>
        <xsl:if test="$isFullHTML='false'">
            <xsl:apply-templates/>
        </xsl:if>
    </xsl:template>

    <!-- Validation Report header -->
    <xsl:template match="report">
        <div>
            <style>
                div {
                font-family: sans-serif;
                }
                .invalid {
                color: red;
                font-weight: bold;
                }
                .valid {
                color: green;
                font-weight: bold;
                }
                th {
                text-align: left;
                }
                .policy th, .policy td {
                padding: 5px;
                }
            </style>
            <script src="https://ajax.googleapis.com/ajax/libs/jquery/2.1.3/jquery.min.js"></script>
            <script type="text/javascript">
                if(typeof $ === 'function'){
                $(document).ready(function(){
                $(".hideable").hide();
                $(".hide-tr").show();
                $(".hide-tr").click(function(){
                $("." + $(this).attr("data-target")).toggle();
                var prevText = $(this).text();
                $(this).text($(this).attr('data-translation-toggle'));
                $(this).attr('data-translation-toggle', prevText)
                return false;
                });
                });
                }
            </script>

            <xsl:variable name="validClass">
                <xsl:choose>
                    <xsl:when
                            test="/report/jobs/job/validationReport/@isCompliant = 'true'">
                        <xsl:value-of select="'valid'"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="'invalid'"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:variable>
            <!-- General information -->
            <table border="0" id="table1">
                <tr>
                    <td width="200">
                        <b>File:</b>
                    </td>
                    <td>
                        <xsl:value-of select="/report/jobs/job/item/name"/>
                    </td>
                </tr>
                <xsl:if test="/report/jobs/job/validationReport/@profileName">
                    <tr>
                        <td width="200">
                            <b>Profile:</b>
                        </td>
                        <td>
                            <xsl:value-of
                                    select="/report/jobs/job/validationReport/@profileName"/>
                        </td>
                    </tr>
                </xsl:if>
<!--                <xsl:if test="/report/jobs/job/validationReport/@isCompliant">-->
<!--                    <tr>-->
<!--                        <td width="200" class="{$validClass}">-->
<!--                            Compliance:-->
<!--                        </td>-->
<!--                        <td class="{$validClass}">-->
<!--                            <xsl:if test="/report/jobs/job/validationReport/@isCompliant = 'true'">-->
<!--                                Passed-->
<!--                            </xsl:if>-->
<!--                            <xsl:if test="/report/jobs/job/validationReport/@isCompliant = 'false'">-->
<!--                                Failed-->
<!--                            </xsl:if>-->
<!--                        </td>-->
<!--                    </tr>-->
<!--                </xsl:if>-->
                <xsl:if test="/report/jobs/job/taskException/exceptionMessage">
                    <tr>
                        <td width="200" class="invalid">
                            Error:
                        </td>
                        <td class="invalid">
                            <xsl:value-of
                                    select="/report/jobs/job/taskException/exceptionMessage"/>
                        </td>
                    </tr>
                </xsl:if>
                <xsl:if test="/report/jobs/job/policyReport">
                    <tr>
                        <xsl:choose>
                            <xsl:when
                                    test="/report/jobs/job/policyReport/@failedChecks > 0">
                                <td class="invalid">Policy compliance:</td>
                                <td class="invalid">Failed</td>
                            </xsl:when>
                            <xsl:otherwise>
                                <td class="valid">Policy compliance:</td>
                                <td class="valid">Passed</td>
                            </xsl:otherwise>
                        </xsl:choose>
                    </tr>
                </xsl:if>
            </table>

            <h2>Statistics</h2>
            <table border="0" id="table2">
                <tr>
                    <td width="250">
                        <b>Version:</b>
                    </td>
                    <td>
                        <xsl:value-of select="/report/buildInformation/releaseDetails[@id=$appName]/@version"/>
                    </td>
                </tr>
                <xsl:if test="$parserType">
                    <tr>
                        <td width="250">
                            <b>Parser:</b>
                        </td>
                        <td>
                            <xsl:value-of
                                    select="$parserType"/>
                        </td>
                    </tr>
                </xsl:if>
                <tr>
                    <td width="250">
                        <b>Build Date:</b>
                    </td>
                    <td>
                        <xsl:value-of select="/report/buildInformation/releaseDetails[@id=$appName]/@buildDate"/>
                    </td>
                </tr>
                <tr>
                    <td width="250">
                        <b>Processing time:</b>
                    </td>
                    <td>
                        <xsl:value-of select="/report/jobs/job/duration"/>
                    </td>
                </tr>
                <xsl:if test="/report/jobs/job/validationReport/details/@passedRules or /report/jobs/job/validationReport/details/@failedRules">
                    <tr>
                        <td width="250">
                            <b>Total rules in Profile:</b>
                        </td>
                        <td>
                            <xsl:value-of
                                    select="/report/jobs/job/validationReport/details/@passedRules + /report/jobs/job/validationReport/details/@failedRules"/>
                        </td>
                    </tr>
                </xsl:if>
                <xsl:if test="/report/jobs/job/validationReport/details/@passedChecks">
                    <tr>
                        <td width="250">
                            <b>Checks:</b>
                        </td>
                        <td>
                            <xsl:value-of
                                    select="/report/jobs/job/validationReport/details/@passedChecks"/>
                        </td>
                    </tr>
                </xsl:if>
                <xsl:if test="/report/jobs/job/validationReport/details/@failedChecks">
                    <tr>
                        <td width="250">
                            <b>Deviations:</b>
                        </td>
                        <td>
                            <xsl:value-of
                                    select="/report/jobs/job/validationReport/details/@failedChecks"/>
                        </td>
                    </tr>
                </xsl:if>
                <xsl:if test="/report/jobs/job/policyReport">
                    <tr>
                        <td width="250">
                            <b>Failed Policy Checks:</b>
                        </td>
                        <td>
                            <xsl:value-of
                                    select="/report/jobs/job/policyReport/@failedChecks"/>
                        </td>
                    </tr>
                </xsl:if>
                <xsl:apply-templates
                        select="/report/jobs/job/metadataRepairReport/@status"/>
                <xsl:apply-templates
                        select="/report/jobs/job/metadataRepairReport/@fixCount"/>
                <xsl:apply-templates
                        select="/report/jobs/job/logs/@logsCount"/>
            </table>
            <xsl:if test="/report/jobs/job/metadataRepairReport/fixes/fix">
                <h2>Metadata fixes information</h2>

                <table border="0" id="table4">
                    <tr style="BACKGROUND: #bcbad6">
                        <td width="800">
                            <b>Fixes</b>
                        </td>
                        <td width="50">
                            <a id="lable{fixesId}" href="#"
                               style="display: none;"
                               class="hide-tr"
                               data-target="hide{fixesId}"
                               data-translation-toggle="Hide">Show
                            </a>
                        </td>
                    </tr>

                    <xsl:apply-templates
                            select="/report/jobs/job/metadataRepairReport/fixes/fix"/>

                </table>
            </xsl:if>

            <xsl:if test="/report/jobs/job/metadataRepairReport/errors/error">
                <h2>Metadata fixer errors information</h2>

                <table border="0" id="table5">
                    <tr style="BACKGROUND: #bcbad6">
                        <td width="800">
                            <b>Fixes</b>
                        </td>
                        <td width="50">
                            <a id="lable{fixererrorsId}" href="#"
                               style="display: none;"
                               class="hide-tr"
                               data-target="hide{fixererrorsId}"
                               data-translation-toggle="Hide">Show
                            </a>
                        </td>
                    </tr>

                    <xsl:apply-templates
                            select="/report/jobs/job/metadataFixesReport/error"/>

                </table>
            </xsl:if>

            <xsl:if test="/report/jobs/job/validationReport/details/rule">
                <h2>Analysis messages</h2>

                <table border="0" id="table3">
                    <tr style="BACKGROUND: #bcbad6">
                        <td width="800">
                            <b>Rule</b>
                        </td>
                        <td width="50">
                            <b>Status</b>
                        </td>
                    </tr>
                    <xsl:apply-templates
                            select="/report/jobs/job/validationReport/details/rule"/>
                </table>
            </xsl:if>

            <xsl:if test="/report/jobs/job/policyReport">
                <xsl:apply-templates select="/report/jobs/job/policyReport"/>
            </xsl:if>

        </div>

        <xsl:if test="/report/jobs/job/logs">
            <h2>Logs information</h2>

            <table border="0" id="table5">
                <tr style="BACKGROUND: #bcbad6">
                    <td width="100">
                        <b>Type</b>
                    </td>
                    <td width="650">
                        <b>Log message</b>
                    </td>
                    <td width="100">
                        <b>Occurrences</b>
                    </td>
                </tr>

                <xsl:apply-templates
                        select="/report/jobs/job/logs/*"/>
            </table>
        </xsl:if>

        <xsl:if test="/report/jobs/job/featuresReport">
            <h2>Features information</h2>

            <table border="0" id="table4">
                <tr style="BACKGROUND: #bcbad6">
                    <td width="800">
                        <b>Feature</b>
                    </td>
                </tr>
                <xsl:apply-templates
                        select="/report/jobs/job/featuresReport/*"/>
            </table>
        </xsl:if>
    </xsl:template>


    <xsl:template match="/report/jobs/job/metadataRepairReport/@status">
        <tr>
            <td width="250">
                <b>Metadata Fixes Status:</b>
            </td>
            <td>
                <xsl:value-of
                        select="/report/jobs/job/metadataRepairReport/@status"/>
            </td>
        </tr>
    </xsl:template>

    <xsl:template match="/report/jobs/job/metadataRepairReport/@fixCount">
        <tr>
            <td width="250">
                <b>Completed Metadata Fixes:</b>
            </td>
            <td>
                <xsl:value-of
                        select="/report/jobs/job/metadataRepairReport/@fixCount"/>
            </td>
        </tr>
    </xsl:template>

    <xsl:template match="/report/jobs/job/logs/@logsCount">
        <tr>
            <td width="250">
                <b>Number of logs:</b>
            </td>
            <td>
                <xsl:value-of
                        select="/report/jobs/job/logs/@logsCount"/>
            </td>
        </tr>
    </xsl:template>

    <!-- Validation Information -->
    <xsl:template match="/report/jobs/job/validationReport/details/rule">

        <xsl:param name="idWithDots" select="concat(@clause, 't', @testNumber)"/>
        <xsl:param name="id" select="translate($idWithDots, '.', '_')"/>

        <xsl:variable name="wikiLinkEnd">
            <xsl:choose>
                <xsl:when test="starts-with(@specification, 'ISO 19005-1')">
                    <xsl:value-of select="'PDFA-Part-1-rules'"/>
                </xsl:when>
                <xsl:when test="starts-with(@specification, 'ISO 19005-2')">
                    <xsl:value-of select="'PDFA-Parts-2-and-3-rules'"/>
                </xsl:when>
                <xsl:when test="starts-with(@specification, 'ISO 19005-3')">
                    <xsl:value-of select="'PDFA-Parts-2-and-3-rules'"/>
                </xsl:when>
                <xsl:when test="starts-with(@specification, 'ISO 19005-4')">
                    <xsl:value-of select="'PDFA-Part-4-rules'"/>
                </xsl:when>
                <xsl:when test="starts-with(@specification, 'ISO 14289-1')">
                    <xsl:value-of select="'PDFUA-Part-1-rules'"/>
                </xsl:when>
                <xsl:when test="starts-with(@specification, 'ISO 14289-2')">
                    <xsl:value-of select="'PDFUA-Part-2-rules'"/>
                </xsl:when>                
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="tempWikiLink">
            <xsl:choose>
                <xsl:when test="starts-with(@specification, 'ISO 32000')">
                    <xsl:value-of select="'https://github.com/pdf-association/arlington-pdf-model/tree/master/tsv/'"/>
                </xsl:when>
                <xsl:when test="'/' = substring($wikiPath, string-length($wikiPath))">
                    <xsl:value-of select="concat($wikiPath, $wikiLinkEnd)"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="concat($wikiPath, '/', $wikiLinkEnd)"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="wikiLink">
            <xsl:choose>
                <xsl:when test="not(starts-with($tempWikiLink, 'http'))">
                    <xsl:value-of select="concat($tempWikiLink, '.html')"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$tempWikiLink"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="ruleLink">
            <xsl:choose>
                <xsl:when test="starts-with(@specification, 'ISO 32000')">
                    <xsl:variable name="originalObject">
                        <xsl:choose>
                            <xsl:when test="contains(object, 'NameTree')">
                                <xsl:value-of select="substring(substring-before(object, 'NameTree'), 2)"/>
                            </xsl:when>
                            <xsl:when test="contains(object, 'NumberTree')">
                                <xsl:value-of select="substring(substring-before(object, 'NumberTree'), 2)"/>
                            </xsl:when>
                            <xsl:when test="'Entry' = substring(object, string-length(object) - 4)">
                                <xsl:value-of select="substring(object, 2, string-length(object) - 6)"/>
                            </xsl:when>
                            <xsl:when test="'SubArray' = substring(object, string-length(object) - 7)">
                                <xsl:value-of select="substring(object, 2, string-length(object) - 9)"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="substring(object, 2)"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:value-of select="concat($wikiLink, translate(substring(@specification, 10, 3), '-', '.'), '/', $originalObject, '.tsv')"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="concat($wikiLink, '#rule-', translate(@clause, '.', ''), '-', @testNumber)"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="hasLink"
                      select="starts-with(@specification, 'ISO 19005-1') or starts-with(@specification, 'ISO 32000') or
                            starts-with(@specification, 'ISO 19005-2') or starts-with(@specification, 'ISO 19005-3') or
                            starts-with(@specification, 'ISO 19005-4') or starts-with(@specification, 'ISO 14289-1') or 
                            starts-with(@specification, 'ISO 14289-2')"/>
        <xsl:variable name="ruleInformation">
            Specification:
            <xsl:value-of select="@specification"/>,
            Clause:
            <xsl:value-of select="@clause"/>,
            Test number:
            <xsl:value-of select="@testNumber"/>
        </xsl:variable>

        <tr style="BACKGROUND: #dcdaf6">
            <td width="800">
                <xsl:choose>
                    <xsl:when test="$hasLink">
                        <a href="{$ruleLink}">
                            <xsl:copy-of select="$ruleInformation" />
                        </a>
                    </xsl:when>
                    <xsl:otherwise>
                        <b>
                            <xsl:copy-of select="$ruleInformation" />
                        </b>
                    </xsl:otherwise>
                </xsl:choose>
            </td>
            <td/>
        </tr>
        <tr style="BACKGROUND: #dcdaf6">
            <td width="800">
                <xsl:value-of
                        select="description"/>
            </td>
            <td width="50">
                <b>
                    <xsl:if test="@status = 'passed'">
                        <font color="green">
                            <b>Message</b>
                        </font>
                    </xsl:if>
                    <xsl:if test="@status = 'failed'">
                        <font color="red">
                            <b>Deviation</b>
                        </font>
                    </xsl:if>
                </b>
            </td>
        </tr>
        <xsl:if test="@failedChecks > 0">
            <tr style="BACKGROUND: #dcdaf6">
                <td width="800">
                    <xsl:value-of select="@failedChecks"/> occurrences
                </td>
                <td width="50">
                    <xsl:if test="@status = 'failed'">
                        <a id="lable{$id}" href="#" style="display: none;"
                           class="hide-tr"
                           data-target="hide{$id}"
                           data-translation-toggle="Hide">Show
                        </a>
                    </xsl:if>
                </td>
            </tr>
            <tr style="BACKGROUND: #eceaf6" class="hideable hide{$id}">
                <td width="800" style="word-break: break-all">
                    <xsl:value-of select="object"/>
                </td>
                <td/>
            </tr>
            <tr style="BACKGROUND: #eceaf6" class="hideable hide{$id}">
                <td width="800" style="word-break: break-all">
                    <xsl:value-of select="test"/>
                </td>
                <td/>
            </tr>
            <xsl:for-each select="check[@status = 'failed']">
                <tr class="hideable hide{$id}">
                    <td width="800" style="word-break: break-all">
                        <xsl:value-of select="context"/>
                    </td>
                </tr>
                <tr class="hideable hide{$id}">
                    <td width="800" style="word-break: break-all">
                        <xsl:value-of select="errorMessage"/>
                    </td>
                </tr>
            </xsl:for-each>
        </xsl:if>

        <tr>
            <td>
                <br/>
            </td>
            <td>
                <br/>
            </td>
        </tr>

    </xsl:template>
    <!-- Features Information -->
    <xsl:template match="/report/jobs/job/featuresReport/*">

        <tr style="BACKGROUND: #dcdaf6">
            <td width="800">
                <a>
                    <xsl:choose>
                        <xsl:when test="local-name()='informationDict'">
                            Information dictionary
                        </xsl:when>
                        <xsl:when test="local-name()='documentResources'">
                            Document resources
                        </xsl:when>
                        <xsl:when test="local-name()='outlines'">
                            Outlines
                        </xsl:when>
                        <xsl:when test="local-name()='lowLevelInfo'">
                            Low level info
                        </xsl:when>
                        <xsl:when test="local-name()='metadata'">
                            Metadata
                        </xsl:when>
                        <xsl:when test="local-name()='signatures'">
                            Signatures
                        </xsl:when>
                        <xsl:when test="local-name()='embeddedFiles'">
                            Embedded files
                        </xsl:when>
                        <xsl:when test="local-name()='iccProfiles'">
                            ICC profiles
                        </xsl:when>
                        <xsl:when test="local-name()='outputIntents'">
                            Output intents
                        </xsl:when>
                        <xsl:when test="local-name()='annotations'">
                            Annotations
                        </xsl:when>
                        <xsl:when test="local-name()='pages'">
                            Pages
                        </xsl:when>
                        <xsl:when test="local-name()='errors'">
                            Errors
                        </xsl:when>
                    </xsl:choose>
                </a>
            </td>
        </tr>

    </xsl:template>

    <xsl:template match="/report/jobs/job/logs/*">
        <tr style="BACKGROUND: #dcdaf6">
            <td width="100" style="word-break: break-all">
            <b>
                <xsl:choose>
                    <xsl:when test="@level = 'SEVERE'">
                        <font color="red">
                            <b>SEVERE</b>
                        </font>
                    </xsl:when>
                    <xsl:when test="@level = 'WARNING'">
                        <font color="orange">
                            <b>WARNING</b>
                        </font>
                    </xsl:when>
                    <xsl:otherwise>
                        <font color="green">
                            <b><xsl:value-of select="@level"/></b>
                        </font>
                    </xsl:otherwise>
                </xsl:choose>
            </b>
            </td>
            <td width="650" style="word-break: break-all">
                <xsl:value-of select="."/>
            </td>
            <td width="100" style="word-break: break-all">
                <xsl:value-of select="@occurrences"/>
            </td>
        </tr>
    </xsl:template>

    <!-- Metadata fixes information -->
    <xsl:template match="/report/jobs/job/metadataRepairReport/fixes/fix">
        <tr class="hideable hide{fixesId}">
            <td width="800" style="word-break: break-all">
                <xsl:value-of select="."/>
            </td>
        </tr>
    </xsl:template>

    <!-- Metadata fixer errors information -->
    <xsl:template match="/report/jobs/job/metadataRepairReport/errors/error">
        <tr class="hideable hide{fixererrorsId}">
            <td width="800" style="word-break: break-all">
                <xsl:value-of select="."/>
            </td>
        </tr>
    </xsl:template>

    <xsl:template match="/report/jobs/job/policyReport">
        <h2>Policy information</h2>
        <table class="policy">
            <tr style="background: #bcbad6">
                <th class="le">Message</th>
                <th>Status</th>
                <th>Location</th>
            </tr>
            <xsl:for-each select="*/check">
                <xsl:variable name="validClass">
                    <xsl:choose>
                        <xsl:when test="@status = 'passed'">
                            <xsl:value-of select="'valid'"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="'invalid'"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <tr style="background: #dcdaf6">
                    <td>
                        <xsl:value-of select="message"/>
                    </td>
                    <td class="{$validClass}">
                        <xsl:value-of select="@status"/>
                    </td>
                    <td>
                        <xsl:value-of select="@location"/>
                    </td>
                </tr>
            </xsl:for-each>
        </table>
    </xsl:template>

</xsl:stylesheet>
