/**
 * This file is part of veraPDF Validation, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF Validation is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF Validation as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF Validation as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.gf.model.impl.arlington;

import org.verapdf.cos.*;
import org.verapdf.model.alayer.*;
import org.verapdf.gf.model.impl.containers.StaticContainers;
import org.verapdf.tools.StaticResources;
import java.util.*;
import org.verapdf.pd.PDNameTreeNode;
import org.verapdf.as.ASAtom;
import java.util.stream.Collectors;
import org.verapdf.pd.structure.PDNumberTreeNode;

public class GFACryptFilterMap extends GFAObject implements ACryptFilterMap {

	public GFACryptFilterMap(COSBase baseObject, COSBase parentObject, String keyName) {
		super(baseObject, parentObject, keyName, "ACryptFilterMap");
	}

	@Override
	public List<? extends org.verapdf.model.baselayer.Object> getLinkedObjects(String link) {
		switch (link) {
			case "Entries":
				return getEntries();
			case "Identity":
				return getIdentity();
			case "StdCF":
				return getStdCF();
			default:
				return super.getLinkedObjects(link);
		}
	}

	private List<ACryptFilterMapEntry> getEntries() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getEntries1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<ACryptFilterMapEntry> getEntries1_5() {
		List<ACryptFilterMapEntry> list = new LinkedList<>();
		for (ASAtom key : baseObject.getKeySet()) {
			if ("Identity".equals(key.getValue()) || "StdCF".equals(key.getValue())) {
				continue;
			}
			COSObject object = this.baseObject.getKey(key);
			list.add(new GFACryptFilterMapEntry(object != null ? object.get() : null, this.baseObject, this.parentObject, keyName, key.getValue()));
		}
		return Collections.unmodifiableList(list);
	}

	private List<ACryptFilter> getIdentity() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getIdentity1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<ACryptFilter> getIdentity1_5() {
		COSObject object = getIdentityValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<ACryptFilter> list = new ArrayList<>(1);
			list.add(new GFACryptFilter((COSDictionary)object.getDirectBase(), this.baseObject, "Identity"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private List<ACryptFilter> getStdCF() {
		switch (StaticContainers.getFlavour()) {
			case ARLINGTON1_5:
			case ARLINGTON1_6:
			case ARLINGTON1_7:
			case ARLINGTON2_0:
				return getStdCF1_5();
			default:
				return Collections.emptyList();
		}
	}

	private List<ACryptFilter> getStdCF1_5() {
		COSObject object = getStdCFValue();
		if (object == null) {
			return Collections.emptyList();
		}
		if (object.getType() == COSObjType.COS_DICT) {
			List<ACryptFilter> list = new ArrayList<>(1);
			list.add(new GFACryptFilter((COSDictionary)object.getDirectBase(), this.baseObject, "StdCF"));
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	@Override
	public Boolean getcontainsIdentity() {
		return this.baseObject.knownKey(ASAtom.getASAtom("Identity"));
	}

	public COSObject getIdentityValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("Identity"));
		return object;
	}

	@Override
	public Boolean getIdentityHasTypeDictionary() {
		COSObject Identity = getIdentityValue();
		return getHasTypeDictionary(Identity);
	}

	@Override
	public Boolean getcontainsStdCF() {
		return this.baseObject.knownKey(ASAtom.getASAtom("StdCF"));
	}

	public COSObject getStdCFValue() {
		COSObject object = this.baseObject.getKey(ASAtom.getASAtom("StdCF"));
		return object;
	}

	@Override
	public Boolean getStdCFHasTypeDictionary() {
		COSObject StdCF = getStdCFValue();
		return getHasTypeDictionary(StdCF);
	}

	@Override
	public String getStdCFAuthEventNameValue() {
		COSObject StdCF = getStdCFValue();
		if (StdCF == null || !StdCF.getType().isDictionaryBased()) {
			return null;
		}
		return new GFACryptFilter(StdCF.getDirectBase(), null, null).getAuthEventNameValue();
	}

}
