/**
 * This file is part of veraPDF wcag algorithms, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF wcag algorithms is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF wcag algorithms as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF wcag algorithms as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.wcag.algorithms.entities.content;

import java.util.ArrayList;
import java.util.List;

public class TextBlock extends TextInfoChunk {

	private final List<TextLine> textLines = new ArrayList<>();

	public TextBlock() {

	}

	public TextBlock(TextLine line) {
		super(line.getBoundingBox(), line.getFontSize(), line.getBaseLine());
		textLines.add(line);
	}

	public TextBlock(TextBlock block) {
		super(block.getBoundingBox(), block.getFontSize(), block.getBaseLine());
		textLines.addAll(block.getLines());
	}

	public List<TextLine> getLines() {
		return textLines;
	}

	public TextLine getFirstLine() {
		if (textLines.isEmpty()) {
			return null;
		}
		return textLines.get(0);
	}

	public TextLine getLastLine() {
		if (textLines.isEmpty()) {
			return null;
		}
		return textLines.get(textLines.size() - 1);
	}

	public void setLastLine(TextLine lastLine) {
		if (!textLines.isEmpty()) {
			textLines.set(textLines.size() - 1, lastLine);
		} else {
			textLines.add(lastLine);
		}
	}

	public void setFirstLine(TextLine firstLine) {
		if (!textLines.isEmpty()) {
			textLines.set(0, firstLine);
		} else {
			textLines.add(firstLine);
		}
	}

	public TextLine getSecondLine() {
		if (textLines.size() > 1) {
			return textLines.get(1);
		}
		return null;
	}

	public TextLine getPenultLine() {
		if (textLines.size() > 1) {
			return textLines.get(textLines.size() - 2);
		}
		return null;
	}

	public void add(TextLine line) {
		textLines.add(line);
		super.add(line);
	}
	
	public void add(List<TextLine> lines) {
		for (TextLine line : lines) {
			add(line);
		}
	}

	public int getLinesNumber() {
		return textLines.size();
	}

	public boolean isEmpty() {
		return textLines.isEmpty() || textLines.stream().allMatch(TextLine::isEmpty);
	}

	@Override
	public String toString() {
		if (textLines.isEmpty()) {
			return "";
		}
		StringBuilder result = new StringBuilder(textLines.get(0).getValue());
		for (int i = 1; i < textLines.size(); ++i) {
			result.append('\n').append(textLines.get(i).getValue());
		}
		return result.toString();
	}

	@Override
	public int hashCode() {
		int result = super.hashCode();
		result = 31 * result + textLines.size();
		for (TextLine textLine : textLines) {
			result = 31 * result + textLine.hashCode();
		}
		return result;
	}

	@Override
	public boolean equals(Object o) {
		if (!super.equals(o)) {
			return false;
		}
		if (!(o instanceof TextBlock)) {
			return false;
		}
		TextBlock that = (TextBlock) o;
		return this.textLines.equals(that.getLines());
	}
}
